<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://ays-pro.com
 * @since      1.0.0
 *
 * @package    Personal_Dictionary
 * @subpackage Personal_Dictionary/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Personal_Dictionary
 * @subpackage Personal_Dictionary/admin
 * @author     Personal Dictionary Team <info@ays-pro.com>
 */
class Personal_Dictionary_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

    private $user_vocabulary_obj;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of this plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;

        add_filter('set-screen-option', array(__CLASS__, 'set_screen'), 10, 3);

		$per_page_array = array(
            'pd_user_vocabulary_per_page',
		);

		foreach($per_page_array as $option_name){
            add_filter('set_screen_option_'.$option_name, array(__CLASS__, 'set_screen'), 10, 3);
        }
	}

	public function admin_menu_styles(){
        
        echo "<style>
            
            #adminmenu a.toplevel_page_personal-dictionary div.wp-menu-image img {
                width: 25px;
                padding: 5px 0 0;
                transition: .3s ease-in-out;
            }
        </style>";

    }

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles($hook_suffix) {

		wp_enqueue_style($this->plugin_name . '-admin', plugin_dir_url(__FILE__) . 'css/admin.css', array(), $this->version, 'all');

        if (false !== strpos($hook_suffix, "plugins.php")){
            wp_enqueue_style( $this->plugin_name . '-sweetalert-css', PERSONAL_DICTIONARY_PUBLIC_URL . '/css/personal-dictionary-sweetalert2.min.css', array(), $this->version, 'all');
        }

		if (false === strpos($hook_suffix, $this->plugin_name))
            return;
		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Personal_Dictionary_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Personal_Dictionary_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

        if (false !== strpos($hook_suffix, $this->plugin_name . "-our-products")){
            wp_enqueue_style( $this->plugin_name . '-sweetalert-css', PERSONAL_DICTIONARY_PUBLIC_URL . '/css/personal-dictionary-sweetalert2.min.css', array(), $this->version, 'all');
        }

		wp_enqueue_style( $this->plugin_name . '-bootstrap', plugin_dir_url(__FILE__) . 'css/bootstrap.min.css', array(), $this->version, 'all');
		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/personal-dictionary-admin.css', array(), $this->version, 'all' );
		wp_enqueue_style( $this->plugin_name . '-games-page', plugin_dir_url( __FILE__ ) . 'css/personal-dictionary-admin-games-page.css', array(), $this->version, 'all' );
		wp_enqueue_style( $this->plugin_name . '-data-bootstrap', plugin_dir_url(__FILE__) . 'css/dataTables.bootstrap4.min.css', array(), $this->version, 'all');
		wp_enqueue_style($this->plugin_name . '-font-awesome', plugin_dir_url(__FILE__) . '/css/personal-dictionary-font-awesome.min.css', array(), $this->version, 'all');
		wp_enqueue_style($this->plugin_name . '-fonts', plugin_dir_url(__FILE__) . '/css/personal-dictionary-fonts.css', array(), $this->version, 'all');
		wp_enqueue_style( $this->plugin_name . '-select2', PERSONAL_DICTIONARY_PUBLIC_URL . '/css/select2.min.css', array(), $this->version, 'all');

	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts($hook_suffix) {

		if (false !== strpos($hook_suffix, "plugins.php")){
            wp_enqueue_script( $this->plugin_name . '-sweetalert-js', PERSONAL_DICTIONARY_PUBLIC_URL . '/js/personal-dictionary-sweetalert2.all.min.js', array('jquery'), $this->version, true );
            wp_enqueue_script( $this->plugin_name . '-admin', plugin_dir_url(__FILE__) . 'js/admin.js', array( 'jquery' ), $this->version, true );
            wp_localize_script( $this->plugin_name . '-admin', 'PersonalDictionaryAdmin', array( 
            	'ajaxUrl'           	=> admin_url('admin-ajax.php')
            ) );
        }

		if (false === strpos($hook_suffix, $this->plugin_name))
            return;

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Personal_Dictionary_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Personal_Dictionary_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script('jquery');
		wp_enqueue_style('wp-color-picker');
		wp_enqueue_script( $this->plugin_name . '-wp-color-picker-alpha', plugin_dir_url(__FILE__) . 'js/wp-color-picker-alpha.min.js', array('wp-color-picker'), $this->version, true);
		wp_enqueue_script( $this->plugin_name . '-select2', PERSONAL_DICTIONARY_PUBLIC_URL . '/js/select2.min.js', array('jquery'), $this->version, true);
		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/personal-dictionary-admin.js', array( 'jquery' ), $this->version, true );
		wp_enqueue_script( $this->plugin_name . '-datatable-min', PERSONAL_DICTIONARY_PUBLIC_URL . '/js/personal-dictionary-datatable.min.js', array('jquery'), $this->version, true);
		wp_enqueue_script( $this->plugin_name . '-pd-popper', plugin_dir_url(__FILE__) . 'js/popper.min.js', array('jquery'), $this->version, true);
		wp_enqueue_script( $this->plugin_name."-pd-bootstrap", plugin_dir_url( __FILE__ ) . 'js/bootstrap.min.js', array( 'jquery' ), $this->version, true );
		wp_localize_script( $this->plugin_name, 'pdAdminLangObj', array(
            'copied'        => __( 'Copied!', $this->plugin_name),
            'clickForCopy'  => __( 'Click for copy.', $this->plugin_name),
        ) );
		wp_localize_script( $this->plugin_name, 'PersonalDictionaryAdmin', array(
            'addImage'	=> __( 'Add Image', $this->plugin_name ),
            'editImage'	=> __( 'Edit Image', $this->plugin_name ),            
        ) );
		wp_enqueue_script( $this->plugin_name . '-charts-google', plugin_dir_url(__FILE__) . 'js/google-chart.js', array('jquery'), $this->version, true);
		wp_enqueue_script( $this->plugin_name . "-db4.min.js", plugin_dir_url( __FILE__ ) . 'js/dataTables.bootstrap4.min.js', array( 'jquery' ), $this->version, true );

        if (false !== strpos($hook_suffix, $this->plugin_name . "-our-products")){
            wp_enqueue_script( $this->plugin_name . '-sweetalert-js', PERSONAL_DICTIONARY_PUBLIC_URL . '/js/personal-dictionary-sweetalert2.all.min.js', array('jquery'), $this->version, true );
        }

        wp_localize_script($this->plugin_name, 'personalDictionary', array(
			'ajax'           	=> admin_url('admin-ajax.php'), 
            "emptyEmailError"               => __( 'Email field is empty', $this->plugin_name),
            "invalidEmailError"             => __( 'Invalid Email address', $this->plugin_name),
            'selectUser'                    => __( 'Select user', $this->plugin_name),
            'pleaseEnterMore'               => __( "Please enter 1 or more characters", $this->plugin_name ),
            'searching'                     => __( "Searching...", $this->plugin_name ),
            'activated'                     => __( "Activated", $this->plugin_name ),
            'errorMsg'                      => __( "Error", $this->plugin_name ),
            'loadResource'                  => __( "Can't load resource.", $this->plugin_name ),
            'somethingWentWrong'            => __( "Maybe something went wrong.", $this->plugin_name ),            
            'greateJob'                     => __( 'Great job', $this->plugin_name),
            'formMoreDetailed'              => __( 'For more detailed configuration visit', $this->plugin_name),
            'greate'                        => __( 'Great!', $this->plugin_name),
		));

		$color_picker_strings = array(
			'clear'            => __( 'Clear', $this->plugin_name ),
			'clearAriaLabel'   => __( 'Clear color', $this->plugin_name ),
			'defaultString'    => __( 'Default', $this->plugin_name ),
			'defaultAriaLabel' => __( 'Select default color', $this->plugin_name ),
			'pick'             => __( 'Select Color', $this->plugin_name ),
			'defaultLabel'     => __( 'Color value', $this->plugin_name ),
		);
		wp_localize_script( $this->plugin_name . '-wp-color-picker-alpha', 'wpColorPickerL10n', $color_picker_strings );

	}

	public function codemirror_enqueue_scripts($hook) {
        if(strpos($hook, $this->plugin_name) !== false){
            if(function_exists('wp_enqueue_code_editor')){
                $cm_settings['codeEditor'] = wp_enqueue_code_editor(array(
                    'type' => 'text/css',
                    'codemirror' => array(
                        'inputStyle' => 'contenteditable',
                        'theme' => 'cobalt',
                    )
                ));

                wp_enqueue_script('wp-theme-plugin-editor');
                wp_localize_script('wp-theme-plugin-editor', 'cm_settings', $cm_settings);

                wp_enqueue_style('wp-codemirror');
            }
        }
    }

	//==== ADD MENU ====
	// Main Menu
	public function add_plugin_admin_menu(){

        /*
         * Add a settings page for this plugin to the Settings menu.
         *
         * NOTE:  Alternative menu locations are available via WordPress administration menu functions.
         *
         *        Administration Menus: http://codex.wordpress.org/Administration_Menus
         *
         */
        $hook_dictionary = add_menu_page(
            __('Dictionary', $this->plugin_name),
            __('Dictionary', $this->plugin_name),
            'manage_options',
            $this->plugin_name,
            array($this, 'display_plugin_pd_main_page'),
            PERSONAL_DICTIONARY_ADMIN_URL . '/images/icons/logo-admin-32x32.png',
            6
        );
		add_action( "load-$hook_dictionary", array( $this, 'add_tabs' ));
    }
	// Main submenu
	public function add_plugin_pd_submenu(){
		$hook_home = add_submenu_page(
            $this->plugin_name,
            __('Home', $this->plugin_name),
            __('Home', $this->plugin_name),
            'manage_options',
            $this->plugin_name,
            array($this, 'display_plugin_pd_main_page')
        );
		add_action( "load-$hook_home", array( $this, 'add_tabs' ));
    }
	// Reports submenu
	public function add_plugin_pd_reports_submenu(){
        $hook_reports = add_submenu_page(
            $this->plugin_name,
            __('Reports', $this->plugin_name),
            __('Reports', $this->plugin_name),
            'manage_options',
            $this->plugin_name."-dictionary-reports",
            array($this, 'display_plugin_pd_reports_page')
        );
		add_action( "load-$hook_reports", array( $this, 'add_tabs' ));

        $hook_user_vocabulary = add_submenu_page(
            'user_vocabulary_slug',
            __('Reports', $this->plugin_name),
            null,
            'manage_options',
            $this->plugin_name."-dictionary-user-vocabulary",
            array($this, 'display_plugin_pd_user_vocabulary_page')
        );

        add_action("load-$hook_user_vocabulary", array($this, 'screen_option_user_vocabulary'));
		
        add_filter('parent_file', array($this,'personal_dictionary_select_submenu'));

    }
    
	// Leaderboard submenu
	public function add_plugin_pd_leaderboard_submenu(){
		$hook_leaderboard = add_submenu_page(
			$this->plugin_name,
			__('Leaderboard', $this->plugin_name),
			__('Leaderboard', $this->plugin_name),
			'manage_options',
			$this->plugin_name."-dictionary-leaderboard",
			array($this, 'display_plugin_pd_leaderboard_page')
		);
		add_action( "load-$hook_leaderboard", array( $this, 'add_tabs' ));
	}

	// Settings submenu
	public function add_plugin_pd_settings_submenu(){
        $hook_settings = add_submenu_page(
            $this->plugin_name,
            __('Settings', $this->plugin_name),
            __('Settings', $this->plugin_name),
            'manage_options',
            $this->plugin_name."-dictionary-settings",
            array($this, 'display_plugin_pd_settings_page')
        );
		add_action( "load-$hook_settings", array( $this, 'add_tabs' ));
    }

	// Games submenu
	public function add_plugin_pd_games_submenu(){
        $hook_games = add_submenu_page(
            $this->plugin_name,
            __('Games', $this->plugin_name),
            __('Games', $this->plugin_name),
            'manage_options',
            $this->plugin_name."-dictionary-games",
            array($this, 'display_plugin_pd_games_page')
        );
		add_action( "load-$hook_games", array( $this, 'add_tabs' ));
    }

    // How to use submenu
	public function add_plugin_pd_how_to_use_submenu(){
		$hook_howtouse = add_submenu_page(
			$this->plugin_name,
			__('How to use', $this->plugin_name),
			__('How to use', $this->plugin_name),
			'manage_options',
			$this->plugin_name."-how-to-use",
			array($this, 'display_plugin_pd_how_to_use_page')
		);
		add_action( "load-$hook_howtouse", array( $this, 'add_tabs' ));
	}

	// Our products submenu
	public function add_plugin_pd_our_products_submenu(){
		$hook_ourproducts = add_submenu_page(
            $this->plugin_name,
            __('Our products', $this->plugin_name),
            __('Our products', $this->plugin_name),
            'manage_options',
            $this->plugin_name."-our-products",
            array($this, 'display_plugin_pd_our_products_page')
        );
		add_action( "load-$hook_ourproducts", array( $this, 'add_tabs' ));
    }

	public function add_tabs() {
        $screen = get_current_screen();
    
        if ( ! $screen) {
            return;
        }
    
        $screen->add_help_tab(
            array(
                'id'      => 'pd_help_tab',
                'title'   => __( 'General Information:
                    '),
                'content' =>
                    '<h2>' . __( 'Personal Dictionary Information', $this->plugin_name) . '</h2>' .
                    '<p>' .
                        __( 'Personal Dictionary plugin allows your users to create and organize their own vocabulary lists, study and memorize the words. This dictionary builder is a must-have tool for online teachers, university professors, language tutors, or any other online educational-related instructors. The tool can be considered an integral part of any learning management. This online vocabulary can make the study activity more organized and efficient. No limit on the count of words and groups you can make! Incredible, isn’t it?',  $this->plugin_name ).'</p>'
            )
        );
    
        $screen->set_help_sidebar(
            '<p><strong>' . __( 'For more information:', $this->plugin_name) . '</strong></p>' .
            '<p>
                <a href="https://www.youtube.com/watch?v=bNqb6LW0S0U" target="_blank">' . __( 'Youtube video tutorials' , $this->plugin_name ) . '</a>
            </p>' .
            '<p>
                <a href="https://ays-pro.com/wordpress-personal-dictionary-user-manual" target="_blank">' . __( 'Documentation', $this->plugin_name ) . '</a>
            </p>'
        );
    }

    // How to use page
	public function display_plugin_pd_how_to_use_page(){
		include_once('partials/how-to-use/personal-dictionary-how-to-use-display.php');
	}

    public function add_plugin_row_meta($meta, $file) {
		if ($file == PERSONAL_DICTIONARY_BASENAME) {
			$meta[] = '<a href="https://wordpress.org/support/plugin/personal-dictionary/" target="_blank">' . esc_html__( 'Free Support', $this->plugin_name ) . '</a>';
		}

		return $meta;
	}

	//==== ADD PAGES ====
	// Main page
	public function display_plugin_pd_main_page(){
		include_once('partials/dictionaries/personal-dictionary-dictionaries-display.php');
    }
	// Reports page
	public function display_plugin_pd_reports_page(){        
		include_once('partials/reports/personal-dictionary-reports-display.php');
    }
	// User vocabulary page
	public function display_plugin_pd_user_vocabulary_page(){        
		include_once('partials/reports/personal-dictionary-user-vocabulary-display.php');
    }
	// Leaderboard page
	public function display_plugin_pd_leaderboard_page(){        
		include_once('partials/leaderboards/personal-dictionary-leaderboards-display.php');
    }
	// Settings page
	public function display_plugin_pd_settings_page(){
		include_once('partials/settings/personal-dictionary-settings-display.php');
    }

	// Settings page
	public function display_plugin_pd_games_page(){
		include_once('partials/games/personal-dictionary-games-display.php');
    }
	// Our products page
	public function display_plugin_pd_our_products_page(){
		include_once('partials/features/personal-dictionary-products-display.php');
    }
	// PRO features page
	// public function display_plugin_pd_pro_features_page(){
	    // include_once('partials/features/personal-dictionary-pro-features-display.php');
    // }

	// Deactivate plugin from plugins page
	public function deactivate_plugin_option(){
        $request_value = $_REQUEST['upgrade_plugin'];
        $upgrade_option = get_option( 'ays_personal_dictionary_upgrade_plugin', '' );
        if($upgrade_option === ''){
            add_option( 'ays_personal_dictionary_upgrade_plugin', $request_value );
        }else{
            update_option( 'ays_personal_dictionary_upgrade_plugin', $request_value );
        }
        ob_end_clean();
        $ob_get_clean = ob_get_clean();
        echo json_encode( array( 'option' => get_option( 'ays_personal_dictionary_upgrade_plugin', '' ) ) );
        wp_die();
    }

	public function аys_pd_admin_footer($a){
        if(isset($_REQUEST['page'])){
            if(false !== strpos($_REQUEST['page'], $this->plugin_name)){
                ?>
				<div class="ays-pd-footer-support-box">
                    <span class="ays-pd-footer-link-row"><a href="https://wordpress.org/support/plugin/personal-dictionary/" target="_blank"><?php echo __( "Support", $this->plugin_name); ?></a></span>
                    <span class="ays-pd-footer-slash-row">/</span>
                    <span class="ays-pd-footer-link-row"><a href="https://ays-pro.com/wordpress-personal-dictionary-user-manual" target="_blank"><?php echo __( "Docs", $this->plugin_name); ?></a></span>
                    <span class="ays-pd-footer-slash-row">/</span>
                    <span class="ays-pd-footer-link-row"><a href="https://ays-demo.com/personal-dictionary-plugin-survey/" target="_blank"><?php echo __( "Suggest a Feature", $this->plugin_name); ?></a></span>
                </div>
                <p style="font-size:13px;text-align:center;font-style:italic;">
                    <span style="margin-left:0px;margin-right:10px;" class="ays_heart_beat"><i class="ays_fa ays_fa_pd ays_fa_heart animated"></i></span>
                    <span><?php echo __( "If you love our plugin, please do big favor and rate us on", $this->plugin_name); ?></span> 
                    <a target="_blank" href='https://wordpress.org/support/plugin/personal-dictionary/reviews/?rate=5#new-post'>WordPress.org</a>
                    <a target="_blank" class="ays-rated-link" href='https://wordpress.org/support/plugin/personal-dictionary/reviews/'>
                        <span class="ays-dashicons ays-dashicons-star-empty"></span>
                        <span class="ays-dashicons ays-dashicons-star-empty"></span>
                        <span class="ays-dashicons ays-dashicons-star-empty"></span>
                        <span class="ays-dashicons ays-dashicons-star-empty"></span>
                        <span class="ays-dashicons ays-dashicons-star-empty"></span>
                    </a>
                    <span class="ays_heart_beat"><i class="ays_fa ays_fa_pd ays_fa_heart animated"></i></span>
                </p>
            <?php
            }
        }
    }

	public function screen_option_user_vocabulary() {
		$option = 'per_page';
		$args = array(
            'label' => __('User Vocabulary', $this->plugin_name),
            'default' => 20,
            'option' => 'pd_user_vocabulary_per_page'
        );

        add_screen_option($option, $args);
		$this->user_vocabulary_obj = new User_Vocabulary_List_Table($this->plugin_name);
	}

	// show selected submenu on checked subpages
	public function personal_dictionary_select_submenu($file) {
        global $plugin_page;
		if ("personal-dictionary-dictionary-user-vocabulary" == $plugin_page) {
            $plugin_page = $this->plugin_name."-dictionary-reports";
        }

		return $file;
	}

	// set screen options pagination
	public static function set_screen( $status, $option, $value ) {
        return $value;
    }

	public function pd_el_widgets_registered() {
        wp_enqueue_style($this->plugin_name . '-admin', plugin_dir_url(__FILE__) . 'css/admin.css', array(), $this->version, 'all');
        // We check if the Elementor plugin has been installed / activated.
        if ( defined( 'ELEMENTOR_PATH' ) && class_exists( 'Elementor\Widget_Base' ) ) {
            // get our own widgets up and running:
            // copied from widgets-manager.php
            if ( class_exists( 'Elementor\Plugin' ) ) {
                if ( is_callable( 'Elementor\Plugin', 'instance' ) ) {
                    $elementor = Elementor\Plugin::instance();
                    if ( isset( $elementor->widgets_manager ) ) {
						if ( defined( 'ELEMENTOR_VERSION' ) && version_compare( ELEMENTOR_VERSION, '3.5.0', '>=' ) ) {
                            if ( method_exists( $elementor->widgets_manager, 'register' ) ) {
								$widget_file = 'plugins/elementor/personal_dictionary_elementor.php';
								$template_file = locate_template( $widget_file );
								if ( !$template_file || !is_readable( $template_file ) ) {
									$template_file = PERSONAL_DICTIONARY_DIR.'pb_templates/personal_dictionary_elementor.php';
								}
								if ( $template_file && is_readable( $template_file ) ) {
									require_once $template_file;
									Elementor\Plugin::instance()->widgets_manager->register( new Elementor\Widget_PD_Custom_Elementor_Thing() );
								}
							}
						} else {
							if ( method_exists( $elementor->widgets_manager, 'register_widget_type' ) ) {
								$widget_file = 'plugins/elementor/personal_dictionary_elementor.php';
								$template_file = locate_template( $widget_file );
								if ( !$template_file || !is_readable( $template_file ) ) {
									$template_file = PERSONAL_DICTIONARY_DIR.'pb_templates/personal_dictionary_elementor.php';
								}
								if ( $template_file && is_readable( $template_file ) ) {
									require_once $template_file;
									Elementor\Plugin::instance()->widgets_manager->register_widget_type( new Elementor\Widget_PD_Custom_Elementor_Thing() );
								}
							}
						}
                    }
                }
            }
        }
    }
    
	/**
     * Determine if the plugin/addon installations are allowed.
     *
     * @since 1.3.9
     *
     * @param string $type Should be `plugin` or `addon`.
     *
     * @return bool
     */
    public static function ays_pd_can_install( $type ) {

        return self::ays_pd_can_do( 'install', $type );
    }

    /**
     * Determine if the plugin/addon activations are allowed.
     *
     * @since 1.3.9
     *
     * @param string $type Should be `plugin` or `addon`.
     *
     * @return bool
     */
    public static function ays_pd_can_activate( $type ) {

        return self::ays_pd_can_do( 'activate', $type );
    }

    /**
     * Determine if the plugin/addon installations/activations are allowed.
     *
     * @since 1.3.9
     *
     * @param string $what Should be 'activate' or 'install'.
     * @param string $type Should be `plugin` or `addon`.
     *
     * @return bool
     */
    public static function ays_pd_can_do( $what, $type ) {

        if ( ! in_array( $what, array( 'install', 'activate' ), true ) ) {
            return false;
        }

        if ( ! in_array( $type, array( 'plugin', 'addon' ), true ) ) {
            return false;
        }

        $capability = $what . '_plugins';

        if ( ! current_user_can( $capability ) ) {
            return false;
        }

        // Determine whether file modifications are allowed and it is activation permissions checking.
        if ( $what === 'install' && ! wp_is_file_mod_allowed( 'ays_pd_can_install' ) ) {
            return false;
        }

        // All plugin checks are done.
        if ( $type === 'plugin' ) {
            return true;
        }
        return false;
    }

    /**
     * Activate plugin.
     *
     * @since 1.0.0
     * @since 1.3.9 Updated the permissions checking.
     */
    public function ays_pd_activate_plugin() {

        // Run a security check.
        check_ajax_referer( $this->plugin_name . '-install-plugin-nonce', sanitize_key( $_REQUEST['_ajax_nonce'] ) );

        // Check for permissions.
        if ( ! current_user_can( 'activate_plugins' ) ) {
            wp_send_json_error( esc_html__( 'Plugin activation is disabled for you on this site.', $this->plugin_name ) );
        }

        $type = 'addon';

        if ( isset( $_POST['plugin'] ) ) {

            if ( ! empty( $_POST['type'] ) ) {
                $type = sanitize_key( $_POST['type'] );
            }

            $plugin   = sanitize_text_field( wp_unslash( $_POST['plugin'] ) );
            $activate = activate_plugins( $plugin );

            if ( ! is_wp_error( $activate ) ) {
                if ( $type === 'plugin' ) {
                    wp_send_json_success( esc_html__( 'Plugin activated.', $this->plugin_name ) );
                } else {
                        ( esc_html__( 'Addon activated.', $this->plugin_name ) );
                }
            }
        }

        if ( $type === 'plugin' ) {
            wp_send_json_error( esc_html__( 'Could not activate the plugin. Please activate it on the Plugins page.', $this->plugin_name ) );
        }

        wp_send_json_error( esc_html__( 'Could not activate the addon. Please activate it on the Plugins page.', $this->plugin_name ) );
    }

    /**
     * Install addon.
     *
     * @since 1.0.0
     * @since 1.3.9 Updated the permissions checking.
     */
    public function ays_pd_install_plugin() {

        // Run a security check.
        check_ajax_referer( $this->plugin_name . '-install-plugin-nonce', sanitize_key( $_REQUEST['_ajax_nonce'] ) );

        $generic_error = esc_html__( 'There was an error while performing your request.', $this->plugin_name );
        $type          = ! empty( $_POST['type'] ) ? sanitize_key( $_POST['type'] ) : '';

        // Check if new installations are allowed.
        if ( ! self::ays_pd_can_install( $type ) ) {
            wp_send_json_error( $generic_error );
        }

        $error = $type === 'plugin'
            ? esc_html__( 'Could not install the plugin. Please download and install it manually.', $this->plugin_name )
            : "";

        $plugin_url = ! empty( $_POST['plugin'] ) ? esc_url_raw( wp_unslash( $_POST['plugin'] ) ) : '';

        if ( empty( $plugin_url ) ) {
            wp_send_json_error( $error );
        }

        // Prepare variables.
        $url = esc_url_raw(
            add_query_arg(
                [
                    'page' => 'personal-dictionary-our-products',
                ],
                admin_url( 'admin.php' )
            )
        );

        ob_start();
        $creds = request_filesystem_credentials( $url, '', false, false, null );

        // Hide the filesystem credentials form.
        ob_end_clean();

        // Check for file system permissions.
        if ( $creds === false ) {
            wp_send_json_error( $error );
        }
        
        if ( ! WP_Filesystem( $creds ) ) {
            wp_send_json_error( $error );
        }

        /*
         * We do not need any extra credentials if we have gotten this far, so let's install the plugin.
         */
        require_once PERSONAL_DICTIONARY_DIR . 'includes/admin/class-personal-dictionary-upgrader.php';
        require_once PERSONAL_DICTIONARY_DIR . 'includes/admin/class-personal-dictionary-install-skin.php';
        require_once PERSONAL_DICTIONARY_DIR . 'includes/admin/class-personal-dictionary-skin.php';


        // Do not allow WordPress to search/download translations, as this will break JS output.
        remove_action( 'upgrader_process_complete', array( 'Language_Pack_Upgrader', 'async_upgrade' ), 20 );

        // Create the plugin upgrader with our custom skin.
        $installer = new PersonalDictionary\Helpers\PersonalDictionaryPluginSilentUpgrader( new Personal_Dictionary_Install_Skin() );

        // Error check.
        if ( ! method_exists( $installer, 'install' ) ) {
            wp_send_json_error( $error );
        }

        $installer->install( $plugin_url );

        // Flush the cache and return the newly installed plugin basename.
        wp_cache_flush();

        $plugin_basename = $installer->plugin_info();

        if ( empty( $plugin_basename ) ) {
            wp_send_json_error( $error );
        }

        $result = array(
            'msg'          => $generic_error,
            'is_activated' => false,
            'basename'     => $plugin_basename,
        );

        // Check for permissions.
        if ( ! current_user_can( 'activate_plugins' ) ) {
            $result['msg'] = $type === 'plugin' ? esc_html__( 'Plugin installed.', $this->plugin_name ) : "";

            wp_send_json_success( $result );
        }

        // Activate the plugin silently.
        $activated = activate_plugin( $plugin_basename );
        remove_action( 'activated_plugin', array( 'ays_sccp_activation_redirect_method', 'gallery_p_gallery_activation_redirect_method', 'poll_maker_activation_redirect_method' ), 100 );

        if ( ! is_wp_error( $activated ) ) {

            $result['is_activated'] = true;
            $result['msg']          = $type === 'plugin' ? esc_html__( 'Plugin installed and activated.', $this->plugin_name ) : esc_html__( 'Addon installed and activated.', $this->plugin_name );

            wp_send_json_success( $result );
        }

        // Fallback error just in case.
        wp_send_json_error( $result );
    }

    /**
     * List of AM plugins that we propose to install.
     *
     * @since 1.3.9
     *
     * @return array
     */
    protected function pd_get_am_plugins() {
       
        if ( !isset( $_SESSION ) ) {
            session_start();
        }

        $images_url = PERSONAL_DICTIONARY_ADMIN_URL . '/images/icons/';

        $plugin_slug = array(
            'quiz-maker',
            'survey-maker',
            'poll-maker',
            'ays-popup-box',
            'gallery-photo-gallery',
            'secure-copy-content-protection',
            'chart-builder',
            'easy-form',
        );

        $plugin_url_arr = array();
        foreach ($plugin_slug as $key => $slug) {
            if ( isset( $_SESSION['ays_pd_our_product_links'] ) && !empty( $_SESSION['ays_pd_our_product_links'] ) 
                && isset( $_SESSION['ays_pd_our_product_links'][$slug] ) && !empty( $_SESSION['ays_pd_our_product_links'][$slug] ) ) {
                $plugin_url = (isset( $_SESSION['ays_pd_our_product_links'][$slug] ) && $_SESSION['ays_pd_our_product_links'][$slug] != "") ? esc_url( $_SESSION['ays_pd_our_product_links'][$slug] ) : "";
            } else {
                $latest_version = $this->ays_pd_get_latest_plugin_version($slug);
                $plugin_url = 'https://downloads.wordpress.org/plugin/'. $slug .'.zip';
                if ( $latest_version != '' ) {
                    $plugin_url = 'https://downloads.wordpress.org/plugin/'. $slug .'.'. $latest_version .'.zip';
                    $_SESSION['ays_pd_our_product_links'][$slug] = $plugin_url;
                }
            }

            $plugin_url_arr[$slug] = $plugin_url;
        }

        $plugins_array = array(
            'fox-lms/fox-lms.php'        => array(
                'icon'        => $images_url . 'icon-fox-lms-128x128.png',
                'name'        => __( 'Fox LMS', 'quiz-maker' ),
                'desc'        => __( 'Build and manage online courses directly on your WordPress site.', 'quiz-maker' ),
                'desc_hidden' => __( 'With the FoxLMS plugin, you can create, sell, and organize courses, lessons, and quizzes, transforming your website into a dynamic e-learning platform.', 'quiz-maker' ),
                'wporg'       => '',
                'buy_now'     => 'https://foxlms.com/pricing/',
                'url'         => '',
            ),
           'quiz-maker/quiz-maker.php'        => array(
                'icon'        => $images_url . 'icon-quiz-128x128.png',
                'name'        => __( 'Quiz Maker', $this->plugin_name ),
                'desc'        => __( 'With our Quiz Maker plugin it’s easy to make a quiz in a short time.', $this->plugin_name ),
                'desc_hidden' => __( 'You to add images to your quiz, order unlimited questions. Also you can style your quiz to satisfy your visitors.', $this->plugin_name ),
                'wporg'       => 'https://wordpress.org/plugins/quiz-maker/',
                'buy_now'     => 'https://ays-pro.com/wordpress/quiz-maker/',
                'url'         => $plugin_url_arr['quiz-maker'],
            ),
            'survey-maker/survey-maker.php'        => array(
                'icon'        => $images_url . 'icon-survey-128x128.png',
                'name'        => __( 'Survey Maker', $this->plugin_name ),
                'desc'        => __( 'Make amazing online surveys and get real-time feedback quickly and easily.', $this->plugin_name ),
                'desc_hidden' => __( 'Learn what your website visitors want, need, and expect with the help of Survey Maker. Build surveys without limiting your needs.', $this->plugin_name ),
                'wporg'       => 'https://wordpress.org/plugins/survey-maker/',
                'buy_now'     => 'https://ays-pro.com/wordpress/survey-maker',
                'url'         => $plugin_url_arr['survey-maker'],
            ),
            'poll-maker/poll-maker-ays.php'        => array(
                'icon'        => $images_url . 'icon-poll-128x128.png',
                'name'        => __( 'Poll Maker', $this->plugin_name ),
                'desc'        => __( 'Create amazing online polls for your WordPress website super easily.', $this->plugin_name ),
                'desc_hidden' => __( 'Build up various types of polls in a minute and get instant feedback on any topic or product.', $this->plugin_name ),
                'wporg'       => 'https://wordpress.org/plugins/poll-maker/',
                'buy_now'     => 'https://ays-pro.com/wordpress/poll-maker/',
                'url'         => $plugin_url_arr['poll-maker'],
            ),
            'ays-popup-box/ays-pb.php'        => array(
                'icon'        => $images_url . 'icon-popup-128x128.png',
                'name'        => __( 'Popup Box', $this->plugin_name ),
                'desc'        => __( 'Popup everything you want! Create informative and promotional popups all in one plugin.', $this->plugin_name ),
                'desc_hidden' => __( 'Attract your visitors and convert them into email subscribers and paying customers.', $this->plugin_name ),
                'wporg'       => 'https://wordpress.org/plugins/ays-popup-box/',
                'buy_now'     => 'https://ays-pro.com/wordpress/popup-box/',
                'url'         => $plugin_url_arr['ays-popup-box'],
            ),
            'gallery-photo-gallery/gallery-photo-gallery.php'        => array(
                'icon'        => $images_url . 'icon-gallery-128x128.png',
                'name'        => __( 'Gallery Photo Gallery', $this->plugin_name ),
                'desc'        => __( 'Create unlimited galleries and include unlimited images in those galleries.', $this->plugin_name ),
                'desc_hidden' => __( 'Represent images in an attractive way. Attract people with your own single and multiple free galleries from your photo library.', $this->plugin_name ),
                'wporg'       => 'https://wordpress.org/plugins/gallery-photo-gallery/',
                'buy_now'     => 'https://ays-pro.com/wordpress/photo-gallery/',
                'url'         => $plugin_url_arr['gallery-photo-gallery'],
            ),
            'secure-copy-content-protection/secure-copy-content-protection.php'        => array(
                'icon'        => $images_url . 'icon-sccp-128x128.png',
                'name'        => __( 'Secure Copy Content Protection', $this->plugin_name ),
                'desc'        => __( 'Disable the right click, copy paste, content selection and copy shortcut keys on your website.', $this->plugin_name ),
                'desc_hidden' => __( 'Protect web content from being plagiarized. Prevent plagiarism from your website with this easy to use plugin.', $this->plugin_name ),
                'wporg'       => 'https://wordpress.org/plugins/secure-copy-content-protection/',
                'buy_now'     => 'https://ays-pro.com/wordpress/secure-copy-content-protection/',
                'url'         => $plugin_url_arr['secure-copy-content-protection'],
            ),
            'chart-builder/chart-builder.php'        => array(
                'icon'        => $images_url . 'icon-chart-128x128.png',
                'name'        => __( 'Chart Builder', $this->plugin_name ),
                'desc'        => __( 'Chart Builder plugin allows you to create beautiful charts', $this->plugin_name ),
                'desc_hidden' => __( ' and graphs easily and quickly.', $this->plugin_name ),
                'wporg'       => 'https://wordpress.org/plugins/chart-builder/',
                'buy_now'     => 'https://ays-pro.com/wordpress/chart-builder/',
                'url'         => $plugin_url_arr['chart-builder'],
            ),
            'easy-form/easy-form.php'        => array(
                'icon'        => $images_url . 'icon-easyform-128x128.png',
                'name'        => __( 'Easy Form', 'chart-builder' ),
                'desc'        => __( 'Choose the best WordPress form builder plugin. ', 'chart-builder' ),
                'desc_hidden' => __( 'Create contact forms, payment forms, surveys, and many more custom forms. Build forms easily with us.', 'chart-builder' ),
                'wporg'       => 'https://wordpress.org/plugins/easy-form/',
                'buy_now'     => 'https://ays-pro.com/wordpress/easy-form',
                'url'         => $plugin_url_arr['easy-form'],
            ),
        );

        return $plugins_array;
    }

    protected function ays_pd_get_latest_plugin_version( $slug ){

        if ( is_null( $slug ) || empty($slug) ) {
            return "";
        }

        $version_latest = "";

        if ( ! function_exists( 'plugins_api' ) ) {
              require_once( ABSPATH . 'wp-admin/includes/plugin-install.php' );
        }

        // set the arguments to get latest info from repository via API ##
        $args = array(
            'slug' => $slug,
            'fields' => array(
                'version' => true,
            )
        );

        /** Prepare our query */
        $call_api = plugins_api( 'plugin_information', $args );

        /** Check for Errors & Display the results */
        if ( is_wp_error( $call_api ) ) {
            $api_error = $call_api->get_error_message();
        } else {

            //echo $call_api; // everything ##
            if ( ! empty( $call_api->version ) ) {
                $version_latest = $call_api->version;
            }
        }

        return $version_latest;
    }

    /**
     * Get AM plugin data to display in the Addons section of About tab.
     *
     * @since 6.4.0.4
     *
     * @param string $plugin      Plugin slug.
     * @param array  $details     Plugin details.
     * @param array  $all_plugins List of all plugins.
     *
     * @return array
     */
    protected function pd_get_plugin_data( $plugin, $details, $all_plugins ) {

        $have_pro = ( ! empty( $details['pro'] ) && ! empty( $details['pro']['plug'] ) );
        $show_pro = false;

        $plugin_data = array();

        if ( $have_pro ) {
            if ( array_key_exists( $plugin, $all_plugins ) ) {
                if ( is_plugin_active( $plugin ) ) {
                    $show_pro = true;
                }
            }
            if ( array_key_exists( $details['pro']['plug'], $all_plugins ) ) {
                $show_pro = true;
            }
            if ( $show_pro ) {
                $plugin  = $details['pro']['plug'];
                $details = $details['pro'];
            }
        }

        if ( array_key_exists( $plugin, $all_plugins ) ) {
            if ( is_plugin_active( $plugin ) ) {
                // Status text/status.
                $plugin_data['status_class'] = 'status-active';
                $plugin_data['status_text']  = esc_html__( 'Active', $this->plugin_name );
                // Button text/status.
                $plugin_data['action_class'] = $plugin_data['status_class'] . ' ays-pd-card__btn-info disabled';
                $plugin_data['action_text']  = esc_html__( 'Activated', $this->plugin_name );
                $plugin_data['plugin_src']   = esc_attr( $plugin );
            } else {
                // Status text/status.
                $plugin_data['status_class'] = 'status-installed';
                $plugin_data['status_text']  = esc_html__( 'Inactive', $this->plugin_name );
                // Button text/status.
                $plugin_data['action_class'] = $plugin_data['status_class'] . ' ays-pd-card__btn-info';
                $plugin_data['action_text']  = esc_html__( 'Activate', $this->plugin_name );
                $plugin_data['plugin_src']   = esc_attr( $plugin );
            }
        } else {
            // Doesn't exist, install.
            // Status text/status.
            $plugin_data['status_class'] = 'status-missing';

            if ( isset( $details['act'] ) && 'go-to-url' === $details['act'] ) {
                $plugin_data['status_class'] = 'status-go-to-url';
            }
            $plugin_data['status_text'] = esc_html__( 'Not Installed', $this->plugin_name );
            // Button text/status.
            $plugin_data['action_class'] = $plugin_data['status_class'] . ' ays-pd-card__btn-info';
            $plugin_data['action_text']  = esc_html__( 'Install Plugin', $this->plugin_name );
            $plugin_data['plugin_src']   = esc_url( $details['url'] );
        }

        $plugin_data['details'] = $details;

        return $plugin_data;
    }

    /**
     * Display the Addons section of About tab.
     *
     * @since 1.3.9
     */
    public function pd_output_about_addons() {

        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $all_plugins          = get_plugins();
        $am_plugins           = $this->pd_get_am_plugins();
        $can_install_plugins  = self::ays_pd_can_install( 'plugin' );
        $can_activate_plugins = self::ays_pd_can_activate( 'plugin' );

        $content = '';
        $content.= '<div class="ays-pd-cards-block">';
        foreach ( $am_plugins as $plugin => $details ){

            $plugin_data = $this->pd_get_plugin_data( $plugin, $details, $all_plugins );
            $plugin_ready_to_activate = $can_activate_plugins
                && isset( $plugin_data['status_class'] )
                && $plugin_data['status_class'] === 'status-installed';
            $plugin_not_activated     = ! isset( $plugin_data['status_class'] )
                || $plugin_data['status_class'] !== 'status-active';

            $plugin_action_class = ( isset( $plugin_data['action_class'] ) && esc_attr( $plugin_data['action_class'] ) != "" ) ? esc_attr( $plugin_data['action_class'] ) : "";

            $plugin_action_class_disbaled = "";
            if ( strpos($plugin_action_class, 'status-active') !== false ) {
                $plugin_action_class_disbaled = "disbaled='true'";
            }

            $content .= '
                <div class="ays-pd-card">
                    <div class="ays-pd-card__content flexible">
                        <div class="ays-pd-card__content-img-box">
                            <img class="ays-pd-card__img" src="'. esc_url( $plugin_data['details']['icon'] ) .'" alt="'. esc_attr( $plugin_data['details']['name'] ) .'">
                        </div>
                        <div class="ays-pd-card__text-block">
                            <h5 class="ays-pd-card__title">'. esc_html( $plugin_data['details']['name'] ) .'</h5>
                            <p class="ays-pd-card__text">'. wp_kses_post( $plugin_data['details']['desc'] ) .'
                                <span class="ays-pd-card__text-hidden">
                                    '. wp_kses_post( $plugin_data['details']['desc_hidden'] ) .'
                                </span>
                            </p>
                        </div>
                    </div>
                    <div class="ays-pd-card__footer">';
                    if( $plugin != "fox-lms/fox-lms.php" ){
                        if ( $can_install_plugins || $plugin_ready_to_activate || ! $details['wporg'] ) {
                            $content .= '<button class="'. esc_attr( $plugin_data['action_class'] ) .'" data-plugin="'. esc_attr( $plugin_data['plugin_src'] ) .'" data-type="plugin" '. $plugin_action_class_disbaled .'>
                                '. wp_kses_post( $plugin_data['action_text'] ) .'
                            </button>';
                        }
                        elseif ( $plugin_not_activated ) {
                            $content .= '<a href="'. esc_url( $details['wporg'] ) .'" target="_blank" rel="noopener noreferrer">
                                '. esc_html_e( 'WordPress.org', $this->plugin_name ) .'
                                <span aria-hidden="true" class="dashicons dashicons-external"></span>
                            </a>';
                        }
                    }
            $content .='
                        <a target="_blank" href="'. esc_url( $plugin_data['details']['buy_now'] ) .'" class="ays-pd-card__btn-primary">'. __('Buy Now', $this->plugin_name) .'</a>
                    </div>
                </div>';
        }
        $install_plugin_nonce = wp_create_nonce( $this->plugin_name . '-install-plugin-nonce' );
        $content.= '<input type="hidden" id="ays_pd_ajax_install_plugin_nonce" name="ays_pd_ajax_install_plugin_nonce" value="'. $install_plugin_nonce .'">';
        $content.= '</div>';

        echo $content;
    }

}
