<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://ays-pro.com
 * @since      1.0.0
 *
 * @package    Personal_Dictionary
 * @subpackage Personal_Dictionary/public
 */
use PhpOffice\PhpSpreadsheet\IOFactory;
/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Personal_Dictionary
 * @subpackage Personal_Dictionary/public
 * @author     Personal Dictionary Team <info@ays-pro.com>
 */
class Personal_Dictionary_Public {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	private $html_class_prefix = 'ays-pd-';
	private $html_name_prefix = 'ays-pd-';
	private $name_prefix = 'pd_';
	private $options;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of the plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;

		add_shortcode('ayspd_userpage', array($this, 'ays_generate_pd_method'));
	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Personal_Dictionary_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Personal_Dictionary_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */
		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/personal-dictionary-public.css', array(), $this->version, 'all' );
		if ( isset($_GET['ays-pd-tab']) && $_GET['ays-pd-tab'] == 'flashcards' ) {
			wp_enqueue_style( $this->plugin_name . '-flashcards', plugin_dir_url( __FILE__ ) . 'css/personal-dictionary-public-flashcards.css', array(), $this->version, 'all' );
		}

		wp_enqueue_style( $this->plugin_name . '-select2', plugin_dir_url(__FILE__) . 'css/select2.min.css', array(), $this->version, 'all');
	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Personal_Dictionary_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Personal_Dictionary_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */
		wp_enqueue_script("jquery");
		wp_enqueue_script('jquery-ui-sortable');
		wp_enqueue_script("jquery-effects-core");
		wp_enqueue_script( $this->plugin_name . '-touch-punch', plugin_dir_url(__FILE__) . 'js/personal-dictionary-public-touch-punch.min.js', array('jquery'), $this->version, true );
		if ( isset($_GET['ays-pd-tab']) && $_GET['ays-pd-tab'] == 'flashcards' ) {
			wp_enqueue_script( $this->plugin_name . '-google', plugin_dir_url(__FILE__) . '/js/google-chart.js', array( 'jquery' ), $this->version, false );
		}
		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/personal-dictionary-public.js', array( 'jquery' ), $this->version, false );
		wp_enqueue_script( $this->plugin_name . '-plugin', plugin_dir_url( __FILE__ ) . 'js/personal-dictionary-public-plugin.js', array( 'jquery' ), $this->version, false );
		wp_enqueue_script( $this->plugin_name . '-select2', plugin_dir_url(__FILE__) . 'js/select2.min.js', array('jquery'), $this->version, false);
        wp_localize_script( $this->plugin_name, 'aysPersonalDictionaryAjaxPublic', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
			'icons' => array(
				'close_icon' => '<svg xmlns="http://www.w3.org/2000/svg" height="24" viewBox="0 0 24 24" width="24"><path d="M0 0h24v24H0z" fill="none"/><path d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z"/></svg>',
				'edit_icon' => '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 16 16" fill="none">
										<path d="M1.56681 11.5658C1.44504 11.5657 1.32601 11.5296 1.22478 11.4619C1.12354 11.3943 1.04463 11.2981 0.998037 11.1856C0.951442 11.0731 0.939248 10.9493 0.962997 10.8299C0.986745 10.7105 1.04537 10.6008 1.13146 10.5146L10.7541 0.892027C10.8109 0.833215 10.8788 0.786304 10.9539 0.754033C11.0291 0.721761 11.1099 0.704774 11.1916 0.704064C11.2734 0.703353 11.3545 0.718933 11.4302 0.749895C11.5058 0.780856 11.5746 0.826579 11.6324 0.884395C11.6902 0.942211 11.7359 1.01096 11.7669 1.08664C11.7979 1.16231 11.8134 1.2434 11.8127 1.32516C11.812 1.40692 11.795 1.48772 11.7628 1.56285C11.7305 1.63797 11.6836 1.70592 11.6248 1.76272L2.00216 11.3853C1.88208 11.5054 1.72445 11.5658 1.56681 11.5658Z" fill="#0E294A" fill-opacity="0.8"/>
										<path d="M0.61626 16.0002C0.524378 16.0003 0.433644 15.9798 0.350709 15.9403C0.267775 15.9007 0.194743 15.8431 0.136967 15.7717C0.0791917 15.7002 0.038137 15.6167 0.0168128 15.5274C-0.00451143 15.438 -0.00556453 15.345 0.0137308 15.2551L0.965093 10.8207C0.981839 10.7414 1.01407 10.6662 1.05993 10.5994C1.10579 10.5326 1.16439 10.4755 1.23235 10.4314C1.30032 10.3873 1.37633 10.3571 1.45601 10.3424C1.5357 10.3277 1.6175 10.3289 1.69672 10.3459C1.77595 10.3629 1.85104 10.3953 1.91769 10.4414C1.98435 10.4875 2.04125 10.5463 2.08515 10.6144C2.12904 10.6825 2.15907 10.7586 2.17349 10.8383C2.18792 10.918 2.18647 10.9998 2.16923 11.079L1.21787 15.5135C1.18836 15.6512 1.11256 15.7746 1.00308 15.8632C0.893608 15.9517 0.757081 16.0001 0.61626 16.0002ZM5.0501 15.0489C4.92833 15.0488 4.8093 15.0127 4.70807 14.945C4.60683 14.8774 4.52792 14.7812 4.48133 14.6687C4.43473 14.5562 4.42254 14.4324 4.44629 14.313C4.47004 14.1936 4.52866 14.0839 4.61475 13.9977L14.2374 4.37544C14.2942 4.31663 14.3621 4.26972 14.4372 4.23745C14.5124 4.20517 14.5932 4.18819 14.6749 4.18748C14.7567 4.18677 14.8378 4.20235 14.9134 4.23331C14.9891 4.26427 15.0579 4.30999 15.1157 4.36781C15.1735 4.42562 15.2192 4.49438 15.2502 4.57005C15.2812 4.64573 15.2967 4.72681 15.296 4.80857C15.2953 4.89033 15.2783 4.97114 15.2461 5.04626C15.2138 5.12139 15.1669 5.18933 15.1081 5.24614L5.48576 14.8684C5.42861 14.9257 5.36071 14.9712 5.28594 15.0021C5.21117 15.0331 5.13102 15.049 5.0501 15.0489Z" fill="#0E294A" fill-opacity="0.8"/>
										<path d="M0.615412 16.0002C0.46368 15.9996 0.317522 15.943 0.20495 15.8412C0.0923771 15.7395 0.0213089 15.5998 0.00536509 15.4489C-0.0105787 15.298 0.0297234 15.1465 0.118548 15.0235C0.207372 14.9005 0.33847 14.8146 0.486717 14.7822L4.92117 13.8309C5.08063 13.7972 5.24694 13.8281 5.38365 13.9168C5.52036 14.0055 5.61632 14.1448 5.65051 14.3042C5.68469 14.4635 5.65431 14.6299 5.56602 14.7669C5.47773 14.9039 5.33873 15.0003 5.17949 15.035L0.745032 15.9864C0.702453 15.9957 0.658993 16.0003 0.615412 16.0002ZM12.9314 7.16795C12.8505 7.1681 12.7704 7.15223 12.6957 7.12127C12.621 7.0903 12.5531 7.04485 12.496 6.98752L9.01295 3.50443C8.95414 3.44763 8.90723 3.37968 8.87496 3.30456C8.84269 3.22943 8.8257 3.14863 8.82499 3.06687C8.82428 2.98511 8.83986 2.90402 8.87082 2.82835C8.90178 2.75267 8.9475 2.68392 9.00532 2.6261C9.06314 2.56829 9.13189 2.52256 9.20756 2.4916C9.28324 2.46064 9.36432 2.44506 9.44608 2.44577C9.52785 2.44648 9.60865 2.46347 9.68377 2.49574C9.7589 2.52801 9.82685 2.57492 9.88365 2.63374L13.3667 6.11683C13.4528 6.20295 13.5115 6.31265 13.5352 6.43208C13.559 6.55151 13.5468 6.6753 13.5002 6.7878C13.4536 6.9003 13.3747 6.99646 13.2734 7.06413C13.1722 7.13179 13.0532 7.16792 12.9314 7.16795ZM14.6731 5.42655C14.5512 5.42665 14.4321 5.39059 14.3307 5.32293C14.2294 5.25528 14.1504 5.15907 14.1037 5.04649C14.0571 4.93392 14.0449 4.81003 14.0687 4.69052C14.0925 4.57101 14.1512 4.46125 14.2374 4.37513C14.5801 4.03245 14.7688 3.56847 14.7688 3.06908C14.7688 2.5697 14.5801 2.10571 14.2374 1.76304C13.8945 1.42006 13.4305 1.23132 12.9311 1.23132C12.4317 1.23132 11.9677 1.42006 11.625 1.76304C11.5679 1.82023 11.5 1.8656 11.4252 1.89655C11.3505 1.9275 11.2704 1.94343 11.1895 1.94343C11.1087 1.94343 11.0286 1.9275 10.9538 1.89655C10.8791 1.8656 10.8112 1.82023 10.754 1.76304C10.6968 1.70585 10.6515 1.63795 10.6205 1.56323C10.5896 1.48851 10.5736 1.40842 10.5736 1.32754C10.5736 1.24666 10.5896 1.16657 10.6205 1.09184C10.6515 1.01712 10.6968 0.949226 10.754 0.892035C11.3292 0.316599 12.1023 -0.000213623 12.9311 -0.000213623C13.7596 -0.000213623 14.533 0.316599 15.1081 0.892035C15.6836 1.46716 16.0004 2.24026 16.0004 3.06908C16.0004 3.89791 15.6836 4.671 15.1081 5.24613C15.0511 5.30342 14.9833 5.34885 14.9086 5.37981C14.834 5.41078 14.7539 5.42666 14.6731 5.42655Z" fill="#0E294A" fill-opacity="0.8"/>
									</svg>',
				'delete_icon' => '<svg xmlns="http://www.w3.org/2000/svg" height="24" viewBox="0 0 24 24" width="24"><path d="M0 0h24v24H0z" fill="none"/><path d="M6 19c0 1.1.9 2 2 2h8c1.1 0 2-.9 2-2V7H6v12zM19 4h-3.5l-1-1h-5l-1 1H5v2h14V4z"/></svg>',
				'more_icon' => '<svg xmlns="http://www.w3.org/2000/svg" width="4" height="16" viewBox="0 0 4 16" fill="none">
										<path d="M2.00045 3.55558C2.9823 3.55558 3.77824 2.75964 3.77824 1.77779C3.77824 0.795944 2.9823 0 2.00045 0C1.0186 0 0.222656 0.795944 0.222656 1.77779C0.222656 2.75964 1.0186 3.55558 2.00045 3.55558Z" fill="#0E294A" fill-opacity="0.8"/>
										<path d="M2.00045 9.7778C2.9823 9.7778 3.77824 8.98185 3.77824 8.00001C3.77824 7.01816 2.9823 6.22221 2.00045 6.22221C1.0186 6.22221 0.222656 7.01816 0.222656 8.00001C0.222656 8.98185 1.0186 9.7778 2.00045 9.7778Z" fill="#0E294A" fill-opacity="0.8"/>
										<path d="M2.00045 16C2.9823 16 3.77824 15.2041 3.77824 14.2222C3.77824 13.2404 2.9823 12.4444 2.00045 12.4444C1.0186 12.4444 0.222656 13.2404 0.222656 14.2222C0.222656 15.2041 1.0186 16 2.00045 16Z" fill="#0E294A" fill-opacity="0.8"/>
									</svg>',
				// 'open_fs' => '<svg xmlns="http://www.w3.org/2000/svg" height="20" fill="#fff" viewBox="0 0 24 24" width="20" class="open_full_screen"><path d="M0 0h24v24H0z" fill="none"></path><path d="M7 14H5v5h5v-2H7v-3zm-2-4h2V7h3V5H5v5zm12 7h-3v2h5v-5h-2v3zM14 5v2h3v3h2V5h-5z"></path></svg>',
				'open_group_word' => '<img src="'.PERSONAL_DICTIONARY_PUBLIC_URL.'/images/icons/plus.svg">',
			),
        ) );

		$pd = Personal_Dictionary_Data::get_pd_globsettings();
		$this->options = Personal_Dictionary_Data::get_pd_validated_data_from_array( $pd );
		$group_name = $this->options[ $this->name_prefix . 'change_group_name' ];
		$word_name = $this->options[ $this->name_prefix . 'change_word_name' ];
		$translation_name = $this->options[ $this->name_prefix . 'change_translation_name' ];
		if ($group_name === 'Group') {
			$group_name = __('Group', $this->plugin_name);
		}
		if ($word_name === 'Word') {
			$word_name = __('Word', $this->plugin_name);
		}
		if ($translation_name === 'Translation') {
			$translation_name = __('Translation', $this->plugin_name);
		}
		wp_localize_script( $this->plugin_name, 'aysPdLangObj', array(
            'save' => __( 'Save', $this->plugin_name ),
            'saveAndClose' => __( 'Save and close', $this->plugin_name ),
            'settingsMessage' => sprintf(__('Game require 5 or more %ss', $this->plugin_name), $word_name),
            'all' => __( 'All', $this->plugin_name ),
            'result' => __( 'Result', $this->plugin_name ),
            'groups' => sprintf(__('%s', $this->plugin_name), $group_name),
            'games' => __( 'Games', $this->plugin_name ),
			'flashcards' => __( 'Flashcards', $this->plugin_name ),
            'reset' => __( 'Reset', $this->plugin_name ),
            'delete' => __( 'Delete', $this->plugin_name ),
            'word' => sprintf(__('%s', $this->plugin_name), $word_name),
    		'translation' => sprintf(__('%s', $this->plugin_name), $translation_name),
			'audio' => __( 'Audio', $this->plugin_name ),
            'image' => __( 'Image', $this->plugin_name ),
			'audioTitle' => __( 'Choose audio', $this->plugin_name ),
			'imageTitle' => __( 'Choose image', $this->plugin_name ),
            'createFirstGroup' => sprintf(__('Create your first %s', $this->plugin_name), $group_name),
    		'createFirstWord' => sprintf(__('Add your %s', $this->plugin_name), $word_name),
            'from' => __( 'From', $this->plugin_name ),
            'to' => __( 'To', $this->plugin_name ),
            'move' => __( 'Move', $this->plugin_name ),
            'moved_to' => sprintf(__('The %s moved to', $this->plugin_name), $word_name),
			'group' => sprintf(__('%s', $this->plugin_name), $group_name),
			'addWord' => sprintf(__('Add a %s', $this->plugin_name), $word_name),
			'addGroup' => sprintf(__('Add a %s', $this->plugin_name), $group_name),
            'defaultSort' => __( 'Default', $this->plugin_name ),
            'deleteWordConfirm' => sprintf(__('Are you sure you want to delete the %s?', $this->plugin_name), $word_name),
    		'intervalMessage' => sprintf(__('There are not enough %ss in that interval', $this->plugin_name), $word_name),
            'learnedPoints' => __( 'Learned', $this->plugin_name ),
			'cancel' => __( 'Cancel', $this->plugin_name ),
			'continueLearning' =>  __( 'Continue learning', $this->plugin_name ),
			'restart' =>  __( 'Restart', $this->plugin_name ),
			'returnToFlashcardsSettings' => __( 'Return to Flashcards settings', $this->plugin_name ),
			'congratulations' => sprintf(__('Congratulations! You have learned all %ss', $this->plugin_name), $word_name),
			'fullScreenNotAllowed' => __( 'Fullscreen mode is not supported on iOS devices.', $this->plugin_name ),
			'check' => __('Check', $this->plugin_name),
			'selectGroup' => sprintf(__('Select %s', $this->plugin_name), $group_name),
			'searchGroup' => sprintf(__('Search %s', $this->plugin_name), $group_name),
			'search' => __( 'Search', $this->plugin_name ),
			'groupName' => sprintf(__('%s name', $this->plugin_name), $group_name),
			'errorMessageUploadImage' => __('Invalid file type! Please upload a valid image file.', $this->plugin_name),
			'errorMessageUploadAudio' => __('Invalid file type! Please upload a valid audio file.', $this->plugin_name),
        ) );
	}

	public function ays_pd_ajax(){
		global $wpdb;

		$response = array(
			"status" => false
		);
		$function = isset($_REQUEST['function']) ? sanitize_text_field( $_REQUEST['function'] ) : null;

		if($function !== null){

			$response = array();
			if( is_callable( array( $this, $function ) ) ){
				$results = $this->$function();
				$response = array(
					"status" => true,
					"results" => $results,
				);
	            ob_end_clean();
	            $ob_get_clean = ob_get_clean();
				echo json_encode( $response );
				wp_die();
			}
		}


        ob_end_clean();
        $ob_get_clean = ob_get_clean();
		echo json_encode( $response );
		wp_die();
	}

	public function ays_groups_add_ajax(){
		global $wpdb;
		$categories_table 	= esc_sql( $wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'categories' );
		$settings_table 	= esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'settings');

		$group_name 		= (isset($_REQUEST['value']) && $_REQUEST['value'] != '') ? stripslashes(sanitize_text_field( $_REQUEST['value'] )) : null;
		$command			= (isset($_REQUEST['command']) && $_REQUEST['command'] != '') ? sanitize_text_field( $_REQUEST['command'] ) : null;
		$catId				= (isset($_REQUEST['catId']) && $_REQUEST['catId'] != '') ? absint( sanitize_text_field( $_REQUEST['catId'] ) ) : null;
		$unique_id = $this->ays_create_cookies();
		$detect_db_column_type = "%d";
		if (get_current_user_id() > 0) {
			$id_row = 'user_id';
			$user_id = absint(get_current_user_id());
		} else {
			$id_row = 'unique_id';
			$user_id = $unique_id;
			$detect_db_column_type = "%s";
		}
		if ($group_name != null) {

			if($command == null && $catId == null){
				$insert_results = $wpdb->insert(
					$categories_table,
					array(
						$id_row => $user_id,
						'name' => $group_name,
						'date_created' => current_time( 'mysql' ),
					),
					array(
						$detect_db_column_type, // user_id
						'%s', // name
						'%s', // date
					)
				);

				$sql  = "SELECT id FROM ".$settings_table ." WHERE " . $id_row . " = " . $user_id ." ";
				$users = $wpdb->get_row($sql,'ARRAY_A');
				
				if( empty($users) ){

					$insert_results = $wpdb->insert(
						$settings_table,
					array(
						$id_row 		=> $user_id,
						'meta_key' 		=> 'register_date',
						'meta_value' 	=> current_time( 'mysql' ),
						'note'       	=> "",
						'options'    	=> ""
					),
					array(
						$detect_db_column_type,
						'%s',
						'%s',
						'%s', 	
						'%s', 	

						)
					);
				}
				
			}else{
				$insert_results = $wpdb->update(
                    $categories_table,
                    array(
						'name' => $group_name,

                    ),
                    array( 'id' => $catId ),
                    array(
                        '%s', // name
                    ),
                    array( '%d' )
                );
			}


			$response = array(
				"status" => true,
				"added_group" => $insert_results,
			);
            ob_end_clean();
            $ob_get_clean = ob_get_clean();
			echo json_encode( $response );
			wp_die();
		}
	}

	public function ays_words_add_ajax(){
		global $wpdb;
		$words_table 		= esc_sql( $wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'words' );
		$category_id		= (isset($_REQUEST['category_id']) && $_REQUEST['category_id'] != '') ? absint( sanitize_text_field( $_REQUEST['category_id'] ) ) : null;
		$word				= (isset($_REQUEST['word']) && $_REQUEST['word'] != '') ? stripslashes(sanitize_text_field( $_REQUEST['word'] )) : null;
		$translation		= (isset($_REQUEST['translation']) && $_REQUEST['translation'] != '') ? stripslashes(sanitize_text_field( $_REQUEST['translation'] )) : '';
		$command			= (isset($_REQUEST['command']) && $_REQUEST['command'] != '') ? absint( sanitize_text_field( $_REQUEST['command'] ) ) : null;
		$wordId				= (isset($_REQUEST['wordId']) && $_REQUEST['wordId'] != '') ? absint( sanitize_text_field( $_REQUEST['wordId'] ) ) : null;
		$audio              = (isset($_REQUEST['audioID'])) ? sanitize_text_field($_REQUEST['audioID']) : '';
		$image              = (isset($_REQUEST['imageID'])) ? sanitize_text_field($_REQUEST['imageID']) : '';
		$unique_id = $this->ays_create_cookies();
		$detect_db_column_type = "%d";
		if (get_current_user_id() > 0) {
			$id_row = 'user_id';
			$user_id = absint(get_current_user_id());
		} else {
			$id_row = 'unique_id';
			$user_id = $unique_id;
			$detect_db_column_type = "%s";
		}

		if ($word !== null && $category_id !== null) {
			
			if($command == null && $wordId == null){
				$insert_results = $wpdb->insert(
					$words_table,
					array(
						$id_row		 	    => $user_id,
						'category_id'	 	=> $category_id,
						'word'			 	=> $word,
						'translation'    	=> $translation,
						'image'             => $image,
                    	'audio' 			=> $audio,
						'date_created'		=> current_time( 'mysql' ),
						'date_modified'		=> current_time( 'mysql' ),
					),
					array(
						$detect_db_column_type, // user_id
						'%d', // category_id
						'%s', // word
						'%s', // translation
						'%s', // image
                    	'%s', // audio
						'%s', // date_created
						'%s', // date_modified
					)
				);
			} else{
				$current_data = $wpdb->get_row($wpdb->prepare("SELECT image, audio FROM $words_table WHERE id = %d", $wordId), ARRAY_A);
				if (!$image) {
					$image = $current_data['image'];
				}
				if (!$audio) {
					$audio = $current_data['audio'];
				}
				$insert_results = $wpdb->update(
                    $words_table,
                    array(
						'word'			 	=> $word,
						'translation'    	=> $translation,
						'image' 			=> $image, 
                    	'audio' 			=> $audio, 
                        'date_modified'     => current_time( 'mysql' ),

                    ),
                    array( 'id' => $wordId ),
                    array(
                        '%s', // word
                        '%s', // translation
						'%s', // image
                    	'%s', // audio
                        '%s', // date_modified
                    ),
                    array( '%d' )
                );
			}
				
			$response = array(
				"status" => true,
				"added_words" => $insert_results,
			);
            ob_end_clean();
            $ob_get_clean = ob_get_clean();
			echo json_encode( $response );
			wp_die();
		}
	}
	
	public function ays_media_delete_ajax() {
		global $wpdb;
		$words_table = esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'words');
		$response = array("status" => false);
		$wordId = isset($_REQUEST['wordId']) ? absint(sanitize_text_field($_REQUEST['wordId'])) : null;
		$mediaType = isset($_REQUEST['mediaType']) ? sanitize_text_field($_REQUEST['mediaType']) : null;
	
		if (!$wordId || !$mediaType) {
			echo json_encode($response);
			wp_die();
		}
	
		$attachment_id = null;
	
		if ($mediaType === 'image') {
			$attachment_id = $wpdb->get_var($wpdb->prepare(
				"SELECT image FROM $words_table WHERE id = %d",
				$wordId
			));
		} elseif ($mediaType === 'audio') {
			$attachment_id = $wpdb->get_var($wpdb->prepare(
				"SELECT audio FROM $words_table WHERE id = %d",
				$wordId
			));
		}
	
		if ($attachment_id) {
			$mime_type = get_post_mime_type($attachment_id);
	
			if (($mediaType === 'image' && strpos($mime_type, 'image/') === 0) || 
				($mediaType === 'audio' && strpos($mime_type, 'audio/') === 0)) {
				wp_delete_attachment($attachment_id, true);
			} else {
				echo json_encode($response);
				wp_die();
			}
		}
	
		$update_data = array(
			'date_modified' => current_time('mysql'),
		);
	
		if ($mediaType === 'image') {
			$update_data['image'] = ''; 
		} elseif ($mediaType === 'audio') {
			$update_data['audio'] = ''; 
		}
	
		$update_result = $wpdb->update(
			$words_table,
			$update_data,
			array('id' => $wordId),
			array('%s', '%s'),
			array('%d')
		);
	
		if ($update_result !== false) {
			$response['status'] = true;
		}
	
		echo json_encode($response);
		wp_die();
	}
	
	
	
	public function ays_groups_delete_ajax(){
		global $wpdb;
		$categories_table 	= esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'categories');
		$words_table 		= esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'words');
		$catId				= (isset($_REQUEST['catId']) && $_REQUEST['catId'] != '') ? absint( sanitize_text_field( $_REQUEST['catId'] ) ) : null;

		$unique_id = $this->ays_create_cookies();
		if (get_current_user_id() > 0) {
			$id_row = 'user_id';
			$user_id = absint(get_current_user_id());
		} else {
			$id_row = 'unique_id';
			$user_id = "'" . $unique_id . "'";
		}

		if ($catId !== null) {
			$media_ids = $wpdb->get_results($wpdb->prepare(
				"SELECT image, audio FROM $words_table WHERE category_id = %d",
				$catId
			));
			
			if (!empty($media_ids)) {
				foreach ($media_ids as $media) {
					if (!empty($media->image)) {
						wp_delete_attachment($media->image, true); 
					}
					if (!empty($media->audio)) {
						wp_delete_attachment($media->audio, true); 
					}
				}
			}
			$deleted_group = $wpdb->delete(
				$categories_table,
				array( 'id' => $catId ),
				array( '%d' )
			);

			$deleted_words = $wpdb->delete(
				$words_table,
				array( 'category_id' => $catId ),
				array( '%d' )
			);
				
			$response = array(
				"status" => true,
				"deleted_group" => $deleted_group,
				"deleted_words" => $deleted_words,
			);
            ob_end_clean();
            $ob_get_clean = ob_get_clean();
			echo json_encode( $response );
			wp_die();
		}
	}

	public function ays_group_reset_ajax(){
		global $wpdb;

		$words_table 		= esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'words');
		$catId				= (isset($_REQUEST['catId']) && $_REQUEST['catId'] != '') ? absint( sanitize_text_field( $_REQUEST['catId'] ) ) : null;

		$unique_id = $this->ays_create_cookies();
		if (get_current_user_id() > 0) {
			$id_row = 'user_id';
			$user_id = absint(get_current_user_id());
		} else {
			$id_row = 'unique_id';
			$user_id = "'" . $unique_id . "'";
		}
		

		if ($catId !== null) {
			
			$update_result = $wpdb->update(
				$words_table,
				array(
					'point'				=> 0,
					'completed'    		=> 0,
					'percentage'    	=> 0,
					'corrects_count'    => 0,
					'failed_count'    	=> 0,
					'attempts_count'    => 0,
					'date_modified'     => current_time( 'mysql' ),
				),
				array( 'category_id' => $catId ),
				array(
					'%f', // point
					'%d', // completed
					'%f', // percentage
					'%d', // corrects_count
					'%d', // failed_count
					'%d', // attempts_count
					'%s', // date_modified
				),
				array( '%d' )
			);
				
			$response = array(
				"status" => true,
				"update_word" => $update_result,
			);

            ob_end_clean();
            $ob_get_clean = ob_get_clean();
			echo json_encode( $response );
			wp_die();
		}
	}

	public function ays_words_delete_ajax(){
		global $wpdb;
		$words_table 	= esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'words');

		$unique_id = $this->ays_create_cookies();
		if (get_current_user_id() > 0) {
			$id_row = 'user_id';
			$user_id = absint(get_current_user_id());
		} else {
			$id_row = 'unique_id';
			$user_id = "'" . $unique_id . "'";
		}

		$wordId			= (isset($_REQUEST['wordId']) && $_REQUEST['wordId'] != '') ? absint( sanitize_text_field( $_REQUEST['wordId'] ) ) : null;

		if ($wordId !== null) {
			
			$media_data = $wpdb->get_row($wpdb->prepare(
				"SELECT image, audio FROM $words_table WHERE id = %d",
				$wordId
			), ARRAY_A);

			if ($media_data) {
				if (!empty($media_data['image'])) {
					wp_delete_attachment($media_data['image'], true);
				}
	
				if (!empty($media_data['audio'])) {
					wp_delete_attachment($media_data['audio'], true);
				}
			}

			$delete_result = $wpdb->delete(
				$words_table,
				array( 'id' => intval($wordId) ),
				array( '%d' )
			);

				
			$response = array(
				"status" => true,
				"deleted_words" => $delete_result,
			);

            ob_end_clean();
            $ob_get_clean = ob_get_clean();
			echo json_encode( $response );
			wp_die();
		}
	}

	public function ays_word_reset_ajax(){
		global $wpdb;
		$words_table 	= esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'words');

		$unique_id = $this->ays_create_cookies();
		if (get_current_user_id() > 0) {
			$id_row = 'user_id';
			$user_id = absint(get_current_user_id());
		} else {
			$id_row = 'unique_id';
			$user_id = "'" . $unique_id . "'";
		}

		$wordId			= (isset($_REQUEST['wordId']) && $_REQUEST['wordId'] != '') ? absint( sanitize_text_field( $_REQUEST['wordId'] ) ) : null;

		if ($wordId !== null) {
			$update_result = $wpdb->update(
				$words_table,
				array(
					'point'				=> 0,
					'completed'    		=> 0,
					'percentage'    	=> 0,
					'corrects_count'    => 0,
					'failed_count'    	=> 0,
					'attempts_count'    => 0,
					'date_modified'     => current_time( 'mysql' ),
				),
				array( 'id' => $wordId ),
				array(
					'%f', // point
					'%d', // completed
					'%f', // percentage
					'%d', // corrects_count
					'%d', // failed_count
					'%d', // attempts_count
					'%s', // date_modified
				),
				array( '%d' )
			);
				
			$response = array(
				"status" => true,
				"update_word" => $update_result,
			);

            ob_end_clean();
            $ob_get_clean = ob_get_clean();
			echo json_encode( $response );
			wp_die();
		}
	}

	public function ays_show_words_ajax(){
		global $wpdb;

		$unique_id = $this->ays_create_cookies();
		$detect_db_column_type = "%d";
		if (get_current_user_id() > 0) {
			$id_row = 'user_id';
			$user_id = absint(get_current_user_id());
		} else {
			$id_row = 'unique_id';
			$user_id = "'" . $unique_id . "'";
			$detect_db_column_type = "%s";
		}

		$cat_id = (isset($_REQUEST['catId']) && $_REQUEST['catId'] != '') ? absint( sanitize_text_field( $_REQUEST['catId'] ) ) : null;
		$results = array();
		if( $cat_id !== null ){
			$categories_table = esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'categories');
			$words_table = esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'words');
			$sql  = "SELECT * FROM ".$words_table . " WHERE category_id = " . $cat_id ."  AND " . $id_row . " = " . $user_id . " ORDER BY id DESC " ;
			$words = $wpdb->get_results($sql, 'ARRAY_A');
			$sql2 = "SELECT `name` FROM ".$categories_table . " WHERE id = " . $cat_id ." ";
			$group_name = $wpdb->get_var($sql2);
			
			$pd = Personal_Dictionary_Data::get_pd_globsettings();
			$this->options = Personal_Dictionary_Data::get_pd_validated_data_from_array( $pd );
			$learnt_point = $this->options[ $this->name_prefix . 'learned_points' ];

			foreach ($words as $key => $value) {
				$percentage = 0;
				if( floatval( $value['point'] ) > 0){
					$percentage = round( ( floatval( $value['point'] ) / $learnt_point ) * 100, 1 );
				}else{
					$percentage = 0;
				}

				$words[$key]['percentage'] = $percentage;
			}

			$results[] = $words;
			$results[] = $group_name;
		}

		
		return $results;
	}

	public function ays_groups_pd(){
		global $wpdb;

		$unique_id = $this->ays_create_cookies();
		if (get_current_user_id() > 0) {
			$id_row = 'user_id';
			$user_id = absint(get_current_user_id());
		} else {
			$id_row = 'unique_id';
			$user_id = "'" . $unique_id . "'";
		}

		$words_table = esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'words');
		$categories_table = esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'categories');
		$sql = "SELECT * FROM ". $categories_table ." WHERE " . $id_row . " = " . $user_id . "  ORDER BY id DESC";
		$groups = $wpdb->get_results($sql, 'ARRAY_A');

		$pd = Personal_Dictionary_Data::get_pd_globsettings();
		$this->options = Personal_Dictionary_Data::get_pd_validated_data_from_array( $pd );

		$learnt_point = $this->options[ $this->name_prefix . 'learned_points' ];
		foreach ($groups as $key => $value) {
			$percentage = 0;
			$w_count = 0;
			$sql2  = "SELECT SUM( point ) as sum, COUNT(id) as count FROM ".$words_table . " WHERE category_id = " . $value['id'] ."  AND " . $id_row . " = " . $user_id . " ";
			$completed_words = $wpdb->get_row($sql2,'ARRAY_A');
			if($completed_words['count'] > 0){
				$percentage = ( floatval( $completed_words['sum'] ) / ( $learnt_point * absint( $completed_words['count'] ) ) ) * 100;
			}else{
				$percentage = 0;
			}

			$w_count = isset($completed_words['count']) ? absint( $completed_words['count'] ) : 0;
			$groups[$key]['percentage'] = $percentage;
			$groups[$key]['w_count'] = $w_count;
		}
		return $groups;
	}

	public function ays_games_pd(){
		return $this->ays_pd_get_groups();
	}

	public function ays_pd_get_groups() {
		global $wpdb;

		$unique_id = $this->ays_create_cookies();
		if (get_current_user_id() > 0) {
			$id_row = 'user_id';
			$user_id = absint(get_current_user_id());
		} else {
			$id_row = 'unique_id';
			$user_id = "'" . $unique_id . "'";
		}

		$categories_table = esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'categories');
		$words_table 	= esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'words');
		$sql = "SELECT * FROM ".$categories_table ." WHERE " . $id_row . " = " . $user_id . " ORDER BY id DESC";
		$groups = $wpdb->get_results($sql, 'ARRAY_A');

		foreach ($groups as $key => $value) {
			$sql2  = "SELECT COUNT(id) as count FROM ".$words_table . " WHERE category_id = " . $value['id'] ."  AND " . $id_row . " = " . $user_id . " AND completed = 0 AND translation != '' ";
			$group_words_count = $wpdb->get_row($sql2,'ARRAY_A');
			$groups[$key]['words_count'] = $group_words_count['count'];
		}

		return $groups;
	}

	public function ays_import_pd( $import_file ){

		global $wpdb;
		$name_arr = explode('.', $import_file['name']);
		$type     = end($name_arr);
		
		if($type == 'xlsx' || $type == 'XLSX') {

		
			$categories_table 	= esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'categories');
			$words_table 		= esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'words');
			$settings_table 	= esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'settings');
			$questions_lines 	= fopen($import_file['tmp_name'], 'r');
			
			// $user_id = get_current_user_id();
			$unique_id = $this->ays_create_cookies();
			$detect_db_column_type = "%d";
		    if (get_current_user_id() > 0) {
			    $id_row = 'user_id';
			    $user_id = absint(get_current_user_id());
		    } else {
			    $id_row = 'unique_id';
			    $user_id = $unique_id;
			    $detect_db_column_type = "%s";
			}

			$file_name = $import_file['name'];
			
			$file_name = explode('.'.$type,$file_name);

			$category_id = (isset($_POST['ays-pd-import-select']) && $_POST['ays-pd-import-select'] != '') ?  sanitize_text_field($_POST['ays-pd-import-select']) : 'new_group';

			$for_import = array();
			
				
			$ver = (float) phpversion();
			if ($ver >= 5.6) {
				require_once(PERSONAL_DICTIONARY_DIR . 'includes/PHPExcel/vendor/autoload.php');
				$spreadsheet = IOFactory::load($import_file['tmp_name']);
				$sheet_data  = $spreadsheet->getActiveSheet()->toArray("", true, true, true);
				
		
				$firstLine = array_values($sheet_data[1]);

				
				if( $firstLine[0] == 'word' &&  $firstLine[1] == 'translation' ){
					$headers = $sheet_data[1];
					unset($sheet_data[1]);
				}
				$headers = array ('0' => 'word','1'=>'translation');
				

					//walk and array_combine with array_values
					foreach ( $sheet_data as &$row ) {
						
						$values = array_values($row);
						$row = array_combine($headers, $values);			
					}
					$res_data = array_values($sheet_data);
					foreach ( $res_data as $key => &$value ) {
						if($value['word'] == ''){
							continue;
						}
						$word = htmlspecialchars_decode(isset($value['word']) && $value['word'] != '' ? $value['word'] : '', ENT_HTML5);
						$translation = htmlspecialchars_decode(isset($value['translation']) && $value['translation'] != '' ? $value['translation'] : '', ENT_HTML5);
						
						$for_import[] = array(
							'word'               => $word,
							'translation'        => $translation,
							
						);
					}

			}

			if($category_id == 'new_group'){

				$insert_cat = $wpdb->insert(
					$categories_table,
				array(
					$id_row => $user_id,
					'name' => $file_name[0],
					'date_created' => current_time( 'mysql' ),
				),
				array(
					$detect_db_column_type, // user_id
					'%s', // name
					'%s', // date
					
					)
				);
				$category_id = $wpdb->insert_id;


				$sql  = "SELECT id FROM ".$settings_table ." WHERE " . $id_row . " = " . $user_id ." ";
				$users = $wpdb->get_row($sql,'ARRAY_A');
				
				if( empty($users) ){

					$insert_results = $wpdb->insert(
						$settings_table,
					array(
						$id_row 		=> $user_id,
						'meta_key' 		=> 'register_date',
						'meta_value' 	=> current_time( 'mysql' ),
						'note'       	=> "",
						'options'    	=> ""
					),
					array(
						$detect_db_column_type,
						'%s',
						'%s',
						'%s', 	
						'%s', 	

						)
					);
				}
			}
			
			foreach($for_import as $key => $value){
				$words_res = $wpdb->insert(
					$words_table,						
					array(
						$id_row		 	=> $user_id,
						'category_id'	 	=> $category_id,
						'word'			 	=> $value['word'],
						'translation'    	=> $value['translation'],
						'date_created'		=> current_time( 'mysql' ),
						'date_modified'		=> current_time( 'mysql' ),
					),
					array(
						$detect_db_column_type, // user_id
						'%d', // category_id
						'%s', // word
						'%s', // translation
						'%s', // date_created
						'%s', // date_modified
						
						)
				);
			
			}
		}
	}

	public function ays_pd_upload_file() {
		$result = false;
		$file_name = $_FILES['file']['name'];
		$file_tmp_name = $_FILES['file']['tmp_name'];
		$file_type = $_FILES['file']['type'];
		$uploaded_file_path = "";
		$uploaded_file_url = "";
		$attachment_id = ""; 
		if ($file_name && $file_tmp_name) {
			if (strpos($file_type, 'image/') === 0) {
				$upload = wp_upload_bits($file_name, null, file_get_contents($file_tmp_name));
				$uploaded_file_path = $upload['file'];
				$uploaded_file_url = $upload['url'];
				
				if ($uploaded_file_path) {
					$wp_filetype = wp_check_filetype($file_name, null);
					$attachment = array(
						'guid'           => $uploaded_file_url,
						'post_mime_type' => $wp_filetype['type'],
						'post_title'     => sanitize_file_name($file_name),
						'post_content'   => '',
						'post_status'    => 'inherit'
					);
		
					$attachment_id = wp_insert_attachment($attachment, $uploaded_file_path);
					if (!is_wp_error($attachment_id)) {
						require_once(ABSPATH . 'wp-admin/includes/image.php');
						$attachment_metadata = wp_generate_attachment_metadata($attachment_id, $uploaded_file_path);
						wp_update_attachment_metadata($attachment_id, $attachment_metadata);
					}
		
					$result = true;
				}
			} else if (strpos($file_type, 'audio/') === 0) {
				$upload = wp_upload_bits($file_name, null, file_get_contents($file_tmp_name));
				$uploaded_file_path = $upload['file'];
				$uploaded_file_url = $upload['url'];
				
				if ($uploaded_file_path) {
					$wp_filetype = wp_check_filetype($file_name, null);
					$attachment = array(
						'guid'           => $uploaded_file_url,
						'post_mime_type' => $wp_filetype['type'],
						'post_title'     => sanitize_file_name($file_name),
						'post_content'   => '',
						'post_status'    => 'inherit'
					);
		
					$attachment_id = wp_insert_attachment($attachment, $uploaded_file_path);
					$result = true;
				}
			} else{
				return array(
					'status' => false,
					'filePath' => "",
					'fileUrl'  => "",
					'fileType' => $file_type,
					'attachmentId' => ""
				);
			}
		}
	
		return array(
			'status' => $result,
			'filePath' => $uploaded_file_path ? $uploaded_file_path : "",
			'fileUrl'  => $uploaded_file_url ? $uploaded_file_url : "",
			'fileType' => $file_type ? $file_type : "",
			'attachmentId' => $attachment_id ? $attachment_id : ""
		);
	}	

	function get_attachment_details() {
		$response = [];
	
		if (isset($_POST['attachment_image_id']) || isset($_POST['attachment_audio_id'])) {
		
			if (isset($_POST['attachment_image_id']) && is_numeric($_POST['attachment_image_id'])) {
				$attachment_id = intval($_POST['attachment_image_id']);
				$attachment = get_post($attachment_id);
				if ($attachment && strpos($attachment->post_mime_type, 'image/') === 0) {
					$response['image'] = [
						'title' => $attachment->post_title,
						'mime_type' => $attachment->post_mime_type,
						'guid' => $attachment->guid,
					];
				}
			}
			if (isset($_POST['attachment_audio_id']) && is_numeric($_POST['attachment_audio_id'])) {
				$attachment_id = intval($_POST['attachment_audio_id']);
				$attachment = get_post($attachment_id);
				if ($attachment && strpos($attachment->post_mime_type, 'audio/') === 0) {
					$response['audio'] = [
						'title' => $attachment->post_title,
						'mime_type' => $attachment->post_mime_type,
						'guid' => $attachment->guid,
					];
				}
			}
			if (!empty($response)) {
				wp_send_json_success($response);
			}
		}
		wp_send_json_error([]);
	}
	

	public function ays_pd_add_game_results(){
		global $wpdb;
		$reports_table = esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'reports');

		$unique_id = $this->ays_create_cookies();
		$detect_db_column_type = "%d";
		if (get_current_user_id() > 0) {
			$id_row = 'user_id';
			$user_id = absint(get_current_user_id());
		} else {
			$id_row = 'unique_id';
			$user_id = "'" . $unique_id . "'";
			$detect_db_column_type = "%s";
		}

		$game_type = (isset($_REQUEST['gameType']) && $_REQUEST['gameType'] != '') ? ( sanitize_text_field( $_REQUEST['gameType'] ) ) : null;
		if ($game_type == 'write_translation') {
			$ays_pd_words = (isset($_REQUEST['ays-pd-word']) && !empty($_REQUEST['ays-pd-word'])) ? $_REQUEST['ays-pd-word'] : array();
		} else {
			$ays_pd_words = (isset($_REQUEST['ays-pd-word']) && !empty($_REQUEST['ays-pd-word'])) ? array_map( 'sanitize_text_field', $_REQUEST['ays-pd-word'] ) : array();
			$ays_pd_translations = (isset($_REQUEST['ays-pd-translation']) && !empty($_REQUEST['ays-pd-translation'])) ? array_map( 'sanitize_text_field', $_REQUEST['ays-pd-translation'] ) : array();
			$words_str = implode(',',$ays_pd_words);
			$translation_str = implode(',',$ays_pd_translations);
		}
		$groups_ids = (isset($_REQUEST['groupsIds']) && !empty($_REQUEST['groupsIds'])) ? array_map( 'sanitize_text_field', $_REQUEST['groupsIds'] ) : null;
		$groups_ids = implode(',',$groups_ids);

		$ays_pd_matching_words = (isset($_REQUEST['ays-pd-matching-words-hidden']) && !empty($_REQUEST['ays-pd-matching-words-hidden'])) ? array_map('sanitize_text_field', explode('*', $_REQUEST['ays-pd-matching-words-hidden'])) : array();
		$ays_pd_matching_words = array_filter($ays_pd_matching_words);
		$ays_pd_matching_translations = (isset($_REQUEST['ays-pd-matching-translations-hidden']) && !empty($_REQUEST['ays-pd-matching-translations-hidden'])) ? array_map('sanitize_text_field', explode('*', $_REQUEST['ays-pd-matching-translations-hidden'])) : array();
		$ays_pd_matching_translations = array_filter($ays_pd_matching_translations);
		$matching_words_str = implode(',',$ays_pd_matching_words);
		$matching_translations_str = implode(',',$ays_pd_matching_translations);

		$pd = Personal_Dictionary_Data::get_pd_globsettings();
		$this->options = Personal_Dictionary_Data::get_pd_validated_data_from_array( $pd );

		$right_points = (isset($this->options[ $this->name_prefix . 'right_points' ]) && $this->options[ $this->name_prefix . 'right_points' ] != '') ? abs( floatval( $this->options[ $this->name_prefix . 'right_points' ] ) ) : 1;

		switch ($game_type) {
			case 'find_word':
				$score_count = 0;
				$words_count = count($ays_pd_words);
				foreach ($ays_pd_translations as $key => $value) {
					if($key == $value){
						$score_count = $score_count + 1;
					}
				}
				if($words_count == 0 || empty($ays_pd_words)){
					$score = 0;
				}else{
					$score = round( ( $score_count / $words_count ) * 100, 1 );
				}
				$points = $score_count * $right_points;

				$insert_results = $wpdb->insert(
					$reports_table,
					array(
						$id_row		     	=> $user_id,
						'words_ids' 		=> $words_str,
						'categories_ids' 	=> $groups_ids,
						'score' 			=> $score,
						'words_count' 		=> $words_count,
						'game_type' 		=> $game_type,
						'points' 			=> $points,
						'complete_date' 	=> current_time( 'mysql' ),
					),
					array(
						$detect_db_column_type, // user_id
						'%s', // words_ids
						'%s', // categories_ids
						'%s', // score
						'%d', // words_count
						'%s', // game_type
						'%f', // points
						'%s', // complete_date
					)
				);
				break;
			case 'find_translation':
				$score_count = 0;
				$words_count = count($ays_pd_translations);
				foreach ($ays_pd_words as $key => $value) {
					if($key == $value){
						$score_count = $score_count + 1;
					}
				}
				$points = $score_count * $right_points;

				if($words_count == 0 || empty($ays_pd_translations)){
					$score = 0;
				}else{
					$score = round( ( $score_count / $words_count ) * 100, 1 );
				}
	
				$insert_results = $wpdb->insert(
					$reports_table,
					array(
						$id_row		    	=> $user_id,
						'words_ids' 		=> $translation_str,
						'categories_ids' 	=> $groups_ids,
						'score' 			=> $score,
						'words_count' 		=> $words_count,
						'game_type' 		=> $game_type,
						'points' 			=> $points,
						'complete_date' 	=> current_time( 'mysql' ),
					),
					array(
						$detect_db_column_type, // user_id
						'%s', // words_ids
						'%s', // categories_ids
						'%s', // score
						'%d', // words_count
						'%s', // game_type
						'%f', // points
						'%s', // complete_date
					)
				);
				break;
			case 'matching_words':
				$score_count = 0;
				$words_count = count($ays_pd_matching_words);
				for($i = 0; $i < $words_count; $i++) {
					if ($ays_pd_matching_words[$i + 1] == $ays_pd_matching_translations[$i + 1]) {
						$score_count = $score_count + 1;
					}
				}
				$points = $score_count * $right_points;

				if($words_count == 0 || empty($ays_pd_matching_words)){
					$score = 1;
				}else{
					$score = round( ( $score_count / $words_count ) * 100, 1 );
				}

				$insert_results = $wpdb->insert(
					$reports_table,
					array(
						$id_row		    	=> $user_id,
						'words_ids' 		=> $matching_words_str,
						'categories_ids' 	=> $groups_ids,
						'score' 			=> $score,
						'words_count' 		=> $words_count,
						'game_type' 		=> $game_type,
						'points' 			=> $points,
						'complete_date' 	=> current_time( 'mysql' ),
					),
					array(
						$detect_db_column_type, // user_id
						'%s', // words_ids
						'%s', // categories_ids
						'%s', // score
						'%d', // words_count
						'%s', // game_type
						'%f', // points
						'%s', // complete_date
					)
				);
				break;
			case 'write_translation':
				$score_count = 0;
				$words_count = count($ays_pd_words);
				$words_ids = array();
				foreach ($ays_pd_words as $key => $value) {
					$words_ids[] = $key;
					$user_nswer = $value[0];
					$correct_answer = $value[1];
					if($user_nswer == $correct_answer){
						$score_count = $score_count + 1;
					}
				}
				$words_str = implode(',',$words_ids);

				if($words_count == 0 || empty($ays_pd_words)){
					$score = 0;
				}else{
					$score = round( ( $score_count / $words_count ) * 100, 1 );
				}

				$points = $score_count * $right_points;

				$insert_results = $wpdb->insert(
					$reports_table,
					array(
						$id_row		     	=> $user_id,
						'words_ids' 		=> $words_str,
						'categories_ids' 	=> $groups_ids,
						'score' 			=> $score,
						'words_count' 		=> $words_count,
						'game_type' 		=> $game_type,
						'points' 			=> $points,
						'complete_date' 	=> current_time( 'mysql' ),
					),
					array(
						$detect_db_column_type, // user_id
						'%s', // words_ids
						'%s', // categories_ids
						'%s', // score
						'%d', // words_count
						'%s', // game_type
						'%f', // points
						'%s', // complete_date
					)
				);
				break;
			default:
				$score_count = 0;
				$words_count = count($ays_pd_words);
				foreach ($ays_pd_translations as $key => $value) {
					if($key == $value){
						$score_count = $score_count + 1;
					}
				}

				if($ays_pd_matching_words != null) {
					$wordsOrderArr = explode("*",$ays_pd_matching_words);
					$translationsOrderArr = explode("*",$ays_pd_matching_translations);
					
					
					for($i = 0;$i < count($wordsOrderArr);$i++) {
						$words_str .= $wordsOrderArr[$i] . ",";
						for($j = 0;$j < count($translationsOrderArr);$j++) {
							if($wordsOrderArr[$i] !== "" && $wordsOrderArr[$i] === $translationsOrderArr[$j] && $i == $j) {
								$score_count++;
							}
						}
					}
					$words_str = trim($words_str,",");
					$words_count = count($wordsOrderArr) -2;
				}

				$points = $score_count * $right_points;

				if($words_count == 0){
					$score = 0;
				}else{
					$score = round( ( $score_count / $words_count ) * 100, 1 );
				}
	
				$insert_results = $wpdb->insert(
					$reports_table,
					array(
						$id_row		 		=> $user_id,
						'words_ids' 		=> $words_str,
						'categories_ids' 	=> $groups_ids,
						'score' 			=> $score,
						'words_count' 		=> $words_count,
						'game_type' 		=> $game_type,
						'points' 			=> $points,
						'complete_date' 	=> current_time( 'mysql' ),
					),
					array(
						$detect_db_column_type, // user_id
						'%s', // words_ids
						'%s', // categories_ids
						'%s', // score
						'%d', // words_count
						'%s', // game_type
						'%f', // points
						'%s', // complete_date
					)
				);
			break;
		}

		$response = array(
			"status" => true,
			"added_report" => $insert_results,
		);
		ob_end_clean();
		$ob_get_clean = ob_get_clean();
		echo json_encode( $response );
		wp_die();
	}

	public function ays_pd_update_word(){
		global $wpdb;

		$unique_id = $this->ays_create_cookies();
		if (get_current_user_id() > 0) {
			$id_row = 'user_id';
			$user_id = absint(get_current_user_id());
		} else {
			$id_row = 'unique_id';
			$user_id = "'" . $unique_id . "'";
		}

		$word_id = (isset($_REQUEST['wordId']) && $_REQUEST['wordId'] != '') ? absint( sanitize_text_field( $_REQUEST['wordId'] ) ) : null;
		$voted = (isset($_REQUEST['voted']) && $_REQUEST['voted'] != '') ? absint( sanitize_text_field( $_REQUEST['voted'] ) ) : null;
		$answered = (isset($_REQUEST['answered']) && $_REQUEST['answered'] != '') ? sanitize_text_field( $_REQUEST['answered'] ) : null;
		$game_type = (isset($_REQUEST['gameType']) && $_REQUEST['gameType'] != '') ? sanitize_text_field( $_REQUEST['gameType'] ) : null;
		$words_table = esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'words');
		$sql = "SELECT * FROM ".$words_table . " WHERE id = " . $word_id ."  AND " . $id_row . " = " . $user_id . " " ;
		$words = $wpdb->get_row($sql,'ARRAY_A');
		$translation = (isset($words['translation']) && $words['translation'] != '') ? sanitize_text_field( $words['translation'] ) : 0;
		$point = (isset($words['point']) && $words['point'] != '') ? floatval( sanitize_text_field( $words['point'] ) ) : 0;
		$completed = (isset($words['completed']) && $words['completed'] != '') ? absint( sanitize_text_field( $words['completed'] ) ) : 0;
		$percentage = (isset($words['percentage']) && $words['percentage'] != '') ? floatval( sanitize_text_field( $words['percentage'] ) ) : 0;
        $corrects_count = (isset($words['corrects_count']) && $words['corrects_count'] != '') ? absint( sanitize_text_field( $words['corrects_count'] ) ) : 0;
		$failed_count = (isset($words['failed_count']) && $words['failed_count'] != '') ? absint( sanitize_text_field( $words['failed_count'] ) ) : 0;
		$attempts_count = (isset($words['attempts_count']) && $words['attempts_count'] != '') ? absint( sanitize_text_field( $words['attempts_count'] ) ) : 0;
		$response = array();

		$pd = Personal_Dictionary_Data::get_pd_globsettings();
		$this->options = Personal_Dictionary_Data::get_pd_validated_data_from_array( $pd );

		$learnt_point = $this->options[ $this->name_prefix . 'learned_points' ];
		$right_points = $this->options[ $this->name_prefix . 'right_points' ];
		$wrong_points = $this->options[ $this->name_prefix . 'wrong_points' ];

		switch ($game_type) {
			case 'write_translation':
				$user_answer = $answered;
				$correct_answer = $translation;
				break;
			default:
				$user_answer = $voted;
				$correct_answer = $word_id;
				break;
		}

		if ($user_answer !== null && $correct_answer !== null) {
			$attempts_count = $attempts_count + 1;
			if($user_answer == $correct_answer) {
				$corrects_count = $corrects_count + 1;
				$point = $point + $right_points;
			} else {
				$failed_count = $failed_count + 1;
				if( $point > 0 ){
					$point = $point - $wrong_points;
				}
			}
			
			if ($point < 0) {
				$point = 0;
			}
			if ($point >= $learnt_point) {
				$completed = 1;
				$point = $learnt_point;
			}

			if ($learnt_point == 0) {
				$percentage = 0;
			} else {
				$percentage = ($point / $learnt_point) * 100;
			}

			if ($percentage > 100) {
				$percentage = 100;
			}

			$update_result = $wpdb->update(
				$words_table,
				array(
					'point'				=> $point,
					'completed'    		=> $completed,
					'percentage'    	=> $percentage,
					'corrects_count'    => $corrects_count,
					'failed_count'    	=> $failed_count,
					'attempts_count'    => $attempts_count,
					'date_modified'     => current_time( 'mysql' ),
				),
				array( 'id' => $word_id ),
				array(
					'%f', // point
					'%d', // completed
					'%f', // percentage
					'%d', // corrects_count
					'%d', // failed_count
					'%d', // attempts_count
					'%s', // date_modified
				),
				array( '%d' )
			);
			$response = array(
				"status" => true,
				"update_words" => $update_result,
			);
            ob_end_clean();
            $ob_get_clean = ob_get_clean();
			echo json_encode( $response );
			wp_die();
		}

		return $response;
	}

	public function ays_pd_game_find_word(){
		global $wpdb;
		$words_table = esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'words');
		$unique_id = $this->ays_create_cookies();
		if (get_current_user_id() > 0) {
			$id_row = 'user_id';
			$user_id = absint(get_current_user_id());
		} else {
			$id_row = 'unique_id';
			$user_id = "'" . $unique_id . "'";
		}

		$groups_ids = (isset($_REQUEST['groupsIds']) && !empty($_REQUEST['groupsIds'])) ? array_map( 'absint', $_REQUEST['groupsIds'] ) : null;
		$ids = (isset($_REQUEST['ids']) && !empty($_REQUEST['ids'])) ? array_map( 'absint', $_REQUEST['ids'] ) : array();
		$words_count = (isset($_REQUEST['wordsCount']) && $_REQUEST['wordsCount'] != '' ) ? sanitize_text_field( $_REQUEST['wordsCount'] ) : 10;
		$limit_number = (isset($_REQUEST['number']) && $_REQUEST['number'] != '') ? sanitize_text_field( $_REQUEST['number'] ) : 0;
		$words_percentage_from = (isset($_REQUEST['wordsPercentageFrom']) && $_REQUEST['wordsPercentageFrom'] != '') ? intval(sanitize_text_field( $_REQUEST['wordsPercentageFrom'] )) : 0;
		$words_percentage_to = (isset($_REQUEST['wordsPercentageTo']) && $_REQUEST['wordsPercentageTo'] != '') ? intval(sanitize_text_field( $_REQUEST['wordsPercentageTo'] )) : 100;

		if ($words_percentage_from == 0) {
			$sql_word_percentage = "AND (`percentage` BETWEEN " . $words_percentage_from . " AND " . $words_percentage_to . " OR `percentage` IS NULL)";
		} else {
			$sql_word_percentage = "AND (`percentage` BETWEEN " . $words_percentage_from . " AND " . $words_percentage_to . ")";
		}

		$limit_number_min = $limit_number;
		$words = array();
		$result_arr = array();

		$sql = "SELECT COUNT(`id`) FROM " . $words_table . " WHERE `category_id` IN(" . sanitize_text_field( implode( ',', $groups_ids ) ) . ") AND `" . $id_row . "` = " . $user_id . " AND `completed` = 0 AND `translation` != '' " . $sql_word_percentage;
		$completed_count = absint( $wpdb->get_var( $sql ) );
		$hidden_count = (isset($_REQUEST['hiddenCount']) && $_REQUEST['hiddenCount'] != null ) ? absint( sanitize_text_field( $_REQUEST['hiddenCount'] ) ) : $completed_count;

		$sql = "SELECT `id` FROM " . $words_table . " WHERE `category_id` IN(" . sanitize_text_field( implode( ',', $groups_ids ) ) . ") AND `" . $id_row . "` = " . $user_id . " AND `completed` = 1 AND `translation` != ''";
		$completed_res = $wpdb->get_results($sql,'ARRAY_A');

		foreach ($completed_res as $key => $value) {	
			array_push($ids,$value['id']);
		}

		$not_in = '';
		if(!empty($ids)){
			$not_in = " AND `id` NOT IN (" . sanitize_text_field( implode( ',', $ids ) ) . ")";
		}else{
			$not_in = '';
		}

		if ( $groups_ids !== null && ( $limit_number_min != $words_count || $limit_number_min == 'all' ) ) {
			$sql = "SELECT * FROM " . $words_table . " WHERE `category_id` IN(" . sanitize_text_field( implode( ',', $groups_ids ) ) . ") AND `" . $id_row . "` = " . $user_id . " AND `translation` != ''";
			$words = $wpdb->get_results($sql, 'ARRAY_A');
			$sql2 = "SELECT * FROM " . $words_table . " WHERE `category_id` IN(" . sanitize_text_field( implode( ',', $groups_ids ) ) . ") AND `" . $id_row . "` = " . $user_id . $not_in . " AND `translation` != '' " . $sql_word_percentage . " ORDER BY RAND() LIMIT 10";
			$ten_words = $wpdb->get_results($sql2, 'ARRAY_A');
			$limit_number_min = $limit_number + 10;
			
			if($completed_count >= 4){

				$translations_arr = array();
				foreach ($ten_words as $key => $value) {
					$limit_number = $limit_number + 1;
					
					$result_arr[$value['id']]['id'] = $value['id'];
					$result_arr[$value['id']]['word'] = $value['word'];

					$result_arr[$value['id']]['translations'][] = array(
						intval( $value['id'] ),
						$value['translation']
					);

					$result_arr[$value['id']]['correct_answer'] = intval($value['id']);
					$result_arr[$value['id']]['count'] = intval($hidden_count);
					$result_arr[$value['id']]['limitNumber'] = $limit_number_min;
					$result_arr[$value['id']]['dataId'] = $limit_number;

					if (isset($value['image']) && !empty($value['image'])) {
						$image_url = wp_get_attachment_url($value['image']);
						$result_arr[$value['id']]['image'] = $image_url;
					}

					if (isset($value['audio']) && !empty($value['audio'])) {
						$audio_url = wp_get_attachment_url($value['audio']);
						$result_arr[$value['id']]['audio'] = $audio_url;
					}
				}
				
				foreach($words as $k => $v){
					$translations_arr[$v['id']] = $v['translation'];
				}
				
				foreach ($result_arr as $key => $value) {
					$translations_arr2 = $translations_arr;
					unset($translations_arr2[$key]);
					$rand_keys = array_rand($translations_arr2,3);
			
					foreach ($rand_keys as $key2 => $value2) {
						$result_arr[$key]['translations'][] = array( $value2, $translations_arr2[$value2] );
						shuffle($result_arr[$key]['translations']);					
					}
				}
			}
		
		}
		shuffle($result_arr);
		return $result_arr;
	}

	public function ays_pd_game_find_translation(){
		global $wpdb;
		$words_table = esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'words');
		$unique_id = $this->ays_create_cookies();
		if (get_current_user_id() > 0) {
			$id_row = 'user_id';
			$user_id = absint(get_current_user_id());
		} else {
			$id_row = 'unique_id';
			$user_id = "'" . $unique_id . "'";
		}

		$groups_ids = (isset($_REQUEST['groupsIds']) && !empty($_REQUEST['groupsIds'])) ? array_map( 'absint', $_REQUEST['groupsIds'] ) : null;
		$ids = (isset($_REQUEST['ids']) && !empty($_REQUEST['ids'])) ? array_map( 'absint', $_REQUEST['ids'] ) : array();
		$words_count = (isset($_REQUEST['wordsCount']) && $_REQUEST['wordsCount'] != '' ) ? sanitize_text_field( $_REQUEST['wordsCount'] ) : 10;
		$limit_number = (isset($_REQUEST['number']) && $_REQUEST['number'] != '') ? sanitize_text_field( $_REQUEST['number'] ) : 0;
		$words_percentage_from = (isset($_REQUEST['wordsPercentageFrom']) && $_REQUEST['wordsPercentageFrom'] != '') ? intval(sanitize_text_field( $_REQUEST['wordsPercentageFrom'] )) : 0;
		$words_percentage_to = (isset($_REQUEST['wordsPercentageTo']) && $_REQUEST['wordsPercentageTo'] != '') ? intval(sanitize_text_field( $_REQUEST['wordsPercentageTo'] )) : 100;

		if ($words_percentage_from == 0) {
			$sql_word_percentage = "AND (`percentage` BETWEEN " . $words_percentage_from . " AND " . $words_percentage_to . " OR `percentage` IS NULL)";
		} else {
			$sql_word_percentage = "AND (`percentage` BETWEEN " . $words_percentage_from . " AND " . $words_percentage_to . ")";
		}

		$limit_number_min = $limit_number;
		$words = array();
		$result_arr = array();

		$sql = "SELECT COUNT(`id`) FROM " . $words_table . " WHERE `category_id` IN(" . sanitize_text_field( implode( ',', $groups_ids ) ) . ") AND `" . $id_row . "` = " . $user_id . " AND `completed` = 0 AND `translation` != '' " . $sql_word_percentage;
		$completed_count = absint( $wpdb->get_var( $sql ) );
		$hidden_count = (isset($_REQUEST['hiddenCount']) && $_REQUEST['hiddenCount'] != null ) ? absint( sanitize_text_field( $_REQUEST['hiddenCount'] ) ) : $completed_count;

		$sql = "SELECT `id` FROM " . $words_table . " WHERE `category_id` IN(" . sanitize_text_field( implode( ',', $groups_ids ) ) . ") AND `" . $id_row . "` = " . $user_id . " AND `completed` = 1 AND `translation` != ''";
		$completed_res = $wpdb->get_results($sql,'ARRAY_A');

		foreach ($completed_res as $key => $value) {	
			array_push($ids,$value['id']);
		}

		$not_in = '';
		if(!empty($ids)){
			$not_in = " AND `id` NOT IN (" . sanitize_text_field( implode( ',', $ids ) ) . ")";
		}else{
			$not_in = '';
		}

		if ( $groups_ids !== null && ( $limit_number_min != $words_count || $limit_number_min == 'all' ) ) {
			$sql = "SELECT * FROM " . $words_table . " WHERE `category_id` IN(" . sanitize_text_field( implode( ',', $groups_ids ) ) . ") AND `" . $id_row . "` = " . $user_id . " AND `translation` != ''";
			$words = $wpdb->get_results($sql, 'ARRAY_A');
			$sql2 = "SELECT * FROM " . $words_table . " WHERE `category_id` IN(" . sanitize_text_field( implode( ',', $groups_ids ) ) . ") AND `" . $id_row . "` = " . $user_id . $not_in . " AND `translation` != '' " . $sql_word_percentage . " ORDER BY RAND() LIMIT 10";
			$ten_translations = $wpdb->get_results($sql2, 'ARRAY_A');
			$limit_number_min = $limit_number + 10;
			
			if($completed_count >= 4){

				$words_arr = array();
				foreach ($ten_translations as $key => $value) {
					$limit_number = $limit_number + 1;
					
					$result_arr[$value['id']]['id'] =  $value['id'];
					$result_arr[$value['id']]['translation'] = $value['translation'];
					
					$result_arr[$value['id']]['words'][] = array(
						intval( $value['id'] ),
						$value['word']
					);
					
					$result_arr[$value['id']]['correct_answer'] = intval($value['id']);
					$result_arr[$value['id']]['count'] = intval($hidden_count);
					$result_arr[$value['id']]['limitNumber'] = $limit_number_min;
					$result_arr[$value['id']]['dataId'] = $limit_number;
					
					if (isset($value['image']) && !empty($value['image'])) {
						$image_url = wp_get_attachment_url($value['image']);
						$result_arr[$value['id']]['image'] = $image_url;
					}
				}
				
				foreach($words as $k => $v){
					$words_arr[$v['id']] =  $v['word'];
				}
				
				foreach ($result_arr as $key => $value) {
					$words_arr2 = $words_arr;
					unset($words_arr2[$key]);
					$rand_keys = array_rand($words_arr2,3);
			
					foreach ($rand_keys as $key2 => $value2) {
						$result_arr[$key]['words'][] = array($value2,$words_arr2[$value2]);
						shuffle($result_arr[$key]['words']);					
					}
				}
			}
		
		}

		shuffle($result_arr);
		return $result_arr;
	}

	public function ays_pd_game_matching_words() {
		global $wpdb;
		$words_table = esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'words');
		$unique_id = $this->ays_create_cookies();
		if (get_current_user_id() > 0) {
			$id_row = 'user_id';
			$user_id = absint(get_current_user_id());
		} else {
			$id_row = 'unique_id';
			$user_id = "'" . $unique_id . "'";
		}

		$groups_ids = (isset($_REQUEST['groupsIds']) && !empty($_REQUEST['groupsIds'])) ? array_map( 'absint', $_REQUEST['groupsIds'] ) : null;
		$ids = (isset($_REQUEST['ids']) && !empty($_REQUEST['ids'])) ? array_map( 'absint', $_REQUEST['ids'] ) : array();
		$words_count = (isset($_REQUEST['wordsCount']) && $_REQUEST['wordsCount'] != '' ) ? sanitize_text_field( $_REQUEST['wordsCount'] ) : 10;
		$limit_number = (isset($_REQUEST['number']) && $_REQUEST['number'] != '') ? sanitize_text_field( $_REQUEST['number'] ) : 0;
		$words_percentage_from = (isset($_REQUEST['wordsPercentageFrom']) && $_REQUEST['wordsPercentageFrom'] != '') ? intval(sanitize_text_field( $_REQUEST['wordsPercentageFrom'] )) : 0;
		$words_percentage_to = (isset($_REQUEST['wordsPercentageTo']) && $_REQUEST['wordsPercentageTo'] != '') ? intval(sanitize_text_field( $_REQUEST['wordsPercentageTo'] )) : 100;

		if($words_percentage_from == 0){
			$sql_word_percentage = " AND (`percentage` BETWEEN ". $words_percentage_from . " AND ". $words_percentage_to . " OR `percentage` IS NULL)";
		}else{
			$sql_word_percentage = " AND (`percentage` BETWEEN ". $words_percentage_from . " AND ". $words_percentage_to . ")";
		}


		$limit_number_min = $limit_number;
		$words = array();
		$result_arr	= array();
		
		$sql = "SELECT COUNT(`id`) FROM ". $words_table . " WHERE `category_id` IN(" .sanitize_text_field( implode(',', $groups_ids) ). ") AND `" . $id_row . "` = ".$user_id . " AND `completed` = 0 AND `translation` != ''".$sql_word_percentage."" ;
		$completed_count = absint( $wpdb->get_var( $sql ) );
		// $hidden_count = (isset($_REQUEST['hiddenCount']) && $_REQUEST['hiddenCount'] != null ) ? absint( sanitize_text_field( $_REQUEST['hiddenCount'] ) ) : $completed_count;
		
		$sql = "SELECT `id` FROM ".$words_table . " WHERE `category_id` IN(" .sanitize_text_field( implode(',', $groups_ids) ). ") AND " . $id_row . " = ".$user_id . " AND `completed` = 1 AND `translation` != ''" ;
		$completed_res = $wpdb->get_results($sql,'ARRAY_A');

		foreach ($completed_res as $key => $value) {	
			array_push($ids,$value['id']);
		}
	
		// $not_in = '';
		// if(!empty($ids)){
		// 	$not_in = " AND `id` NOT IN (" . sanitize_text_field( implode( ',', $ids ) ) . ")";
		// }else{
		// 	$not_in = '';
		// }

		$limit = "";
		if($words_count != "All"){
			$limit =  "LIMIT " . $words_count;
		}

		
		if($groups_ids !== null && ($limit_number_min != $words_count || $limit_number_min == 'all')){
			
			$sql = "SELECT * FROM " . $words_table . " WHERE `category_id` IN(" . sanitize_text_field( implode( ',', $groups_ids ) ) . ") AND `" . $id_row . "` = " . $user_id . " AND `translation` != '' AND `completed` = 0 " . $sql_word_percentage . " ORDER BY RAND() " . $limit ;
			$words = $wpdb->get_results($sql, 'ARRAY_A');
			
			if($completed_count >= 4){
				foreach ($words as $key => $value) {
					$limit_number = $limit_number + 1;

					$result_arr['translations'][] = array(
						intval( $value['id'] ),
						$value['translation']
					);
					
					$result_arr['words'][] = array(
						intval( $value['id'] ),
						$value['word']
					);

					$result_arr['count'] = intval($limit_number);
				}

				shuffle($result_arr['words']);					
				shuffle($result_arr['translations']);
			}
		}
		return $result_arr;
	}

	public function ays_pd_game_write_translation() {
		global $wpdb;
		$words_table = esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'words');
		$unique_id = $this->ays_create_cookies();
		if (get_current_user_id() > 0) {
			$id_row = 'user_id';
			$user_id = absint(get_current_user_id());
		} else {
			$id_row = 'unique_id';
			$user_id = "'" . $unique_id . "'";
		}

		$groups_ids = (isset($_REQUEST['groupsIds']) && !empty($_REQUEST['groupsIds'])) ? array_map( 'absint', $_REQUEST['groupsIds'] ) : null;
		$ids = (isset($_REQUEST['ids']) && !empty($_REQUEST['ids'])) ? array_map( 'absint', $_REQUEST['ids'] ) : array();
		$words_count = (isset($_REQUEST['wordsCount']) && $_REQUEST['wordsCount'] != '' ) ? sanitize_text_field( $_REQUEST['wordsCount'] ) : 10;
		$limit_number = (isset($_REQUEST['number']) && $_REQUEST['number'] != '') ? sanitize_text_field( $_REQUEST['number'] ) : 0;
		$words_percentage_from = (isset($_REQUEST['wordsPercentageFrom']) && $_REQUEST['wordsPercentageFrom'] != '') ? intval(sanitize_text_field( $_REQUEST['wordsPercentageFrom'] )) : 0;
		$words_percentage_to = (isset($_REQUEST['wordsPercentageTo']) && $_REQUEST['wordsPercentageTo'] != '') ? intval(sanitize_text_field( $_REQUEST['wordsPercentageTo'] )) : 100;

		if ($words_percentage_from == 0) {
			$sql_word_percentage = "AND (`percentage` BETWEEN " . $words_percentage_from . " AND " . $words_percentage_to . " OR `percentage` IS NULL)";
		} else {
			$sql_word_percentage = "AND (`percentage` BETWEEN " . $words_percentage_from . " AND " . $words_percentage_to . ")";
		}

		$limit_number_min = $limit_number;
		$result_arr = array();

		$sql = "SELECT COUNT(`id`) FROM " . $words_table . " WHERE `category_id` IN(" . sanitize_text_field( implode( ',', $groups_ids ) ) . ") AND `" . $id_row . "` = " . $user_id . " AND `completed` = 0 AND `translation` != '' " . $sql_word_percentage;
		$completed_count = absint( $wpdb->get_var( $sql ) );
		$hidden_count = (isset($_REQUEST['hiddenCount']) && $_REQUEST['hiddenCount'] != null ) ? absint( sanitize_text_field( $_REQUEST['hiddenCount'] ) ) : $completed_count;

		$sql = "SELECT `id` FROM " . $words_table . " WHERE `category_id` IN(" . sanitize_text_field( implode( ',', $groups_ids ) ) . ") AND `" . $id_row . "` = " . $user_id . " AND `completed` = 1 AND `translation` != ''";
		$completed_res = $wpdb->get_results($sql,'ARRAY_A');

		foreach ($completed_res as $key => $value) {	
			array_push($ids,$value['id']);
		}

		$not_in = '';
		if(!empty($ids)){
			$not_in = " AND `id` NOT IN (" . sanitize_text_field( implode( ',', $ids ) ) . ")";
		}else{
			$not_in = '';
		}

		if ( $groups_ids !== null && ( $limit_number_min != $words_count || $limit_number_min == 'all' ) ) {
			$sql = "SELECT * FROM " . $words_table . " WHERE `category_id` IN(" . sanitize_text_field( implode( ',', $groups_ids ) ) . ") AND `" . $id_row . "` = " . $user_id . $not_in . " AND `translation` != '' " . $sql_word_percentage . " ORDER BY RAND() LIMIT 10";
			$ten_words = $wpdb->get_results($sql, 'ARRAY_A');
			$limit_number_min = $limit_number + 10;
			
			if($completed_count >= 4){
				foreach ($ten_words as $key => $value) {
					$limit_number = $limit_number + 1;
					if (isset($value['image']) && !empty($value['image'])) {
						$image_url = wp_get_attachment_url($value['image']);	
					} else{
						$image_url="";
					}
					
					if (isset($value['audio']) && !empty($value['audio'])) {
						$audio_url = wp_get_attachment_url($value['audio']);
					} else{
						$audio_url="";
					}
					$result_arr['words'][] = array(
						'id' => intval( $value['id'] ),
						'word' => $value['word'],
						'dataId' => $limit_number,
						'translation' => $value['translation'],
						'image' => $image_url,
						'audio' => $audio_url
					);					
				}

				$result_arr['count'] = intval($hidden_count);
				$result_arr['limitNumber'] = $limit_number_min;
			}
		}

		return $result_arr;
	}

	public function ays_pd_get_data_for_flashcards() {
		global $wpdb;

		$words_table = esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'words');
		$unique_id = $this->ays_create_cookies();
		if (get_current_user_id() > 0) {
			$id_row = 'user_id';
			$user_id = absint(get_current_user_id());
		} else {
			$id_row = 'unique_id';
			$user_id = "'" . $unique_id . "'";
		}

		$groups_ids = (isset($_REQUEST['groupsIds']) && !empty($_REQUEST['groupsIds'])) ? array_map( 'absint', $_REQUEST['groupsIds'] ) : null;
		$words_count = (isset($_REQUEST['wordsCount']) && $_REQUEST['wordsCount'] != '' ) ? sanitize_text_field( $_REQUEST['wordsCount'] ) : 10;

		if ($groups_ids !== null) {
			$sql = "SELECT id, word, translation FROM ".$words_table." WHERE `category_id` IN(".sanitize_text_field( implode(',', $groups_ids) ).") AND `" . $id_row . "` = " . $user_id . " AND completed = 0 AND `translation` != ''";
			$words = $wpdb->get_results($sql, 'ARRAY_A');
			shuffle($words);

			$count = count($words);
			if ($words_count != 'All') {
				// if words less than selected words count, than get all words
				$count = ( count($words) > intval($words_count) ) ? intval($words_count) : count($words);
			}

			$selected_words = array_slice($words, 0, $count);

			return $selected_words;
		}
	}

	public function ays_create_cookies () {
		if (!is_user_logged_in()) {
			if (!isset($_COOKIE['ays_pd_unique_id'])) {
				$unique_id = uniqid();
				setcookie('ays_pd_unique_id', $unique_id, time() + (365 * 24 * 60 * 60));
			} else {
				$unique_id = $_COOKIE['ays_pd_unique_id'];
			}
		} else {
			$unique_id = '';
		}
		
		return $unique_id;
	}

	public function ays_generate_pd_method(){
		ob_start();
        $this->enqueue_styles();
        $this->enqueue_scripts();

		$pd = Personal_Dictionary_Data::get_pd_globsettings();

		$this->options = Personal_Dictionary_Data::get_pd_validated_data_from_array( $pd );

		$custom_class = $this->options[ $this->name_prefix . 'custom_class' ];

		$pd_title = isset($this->options[ $this->name_prefix . 'title' ]) && $this->options[ $this->name_prefix . 'title' ] != '' ?  stripslashes ( esc_attr($this->options[ $this->name_prefix . 'title' ])) : "Dictionary" ;

		$content = array();

		$this->ays_create_cookies();

		$show_pd_description = $this->options[ $this->name_prefix . 'show_description' ];
		$show_description_mobile = $this->options[ $this->name_prefix . 'show_description_mobile' ];

		$menu_position_class = $this->html_class_prefix . 'menu-position-' . $this->options[ $this->name_prefix . 'menu_position' ];
		$menu_position_mobile_class = $this->html_class_prefix . 'menu-position-mobile-' . $this->options[ $this->name_prefix . 'menu_position_mobile' ];

		$content[] = '<h2 class="ays-pd-title" id="ays-pd-title-id">'. $pd_title .'</h2>';
		if($show_pd_description == 'on' || $show_description_mobile == 'on'){
        	$pd_description = isset($this->options[ $this->name_prefix . 'description' ]) && $this->options[ $this->name_prefix . 'description' ] != '' ?  stripslashes ( $this->options[ $this->name_prefix . 'description' ] ) : "" ;
			$pd_description = Personal_Dictionary_Data::ays_autoembed( $pd_description );
        }else{
            $pd_description = "";
        }
		$content[] = '<div class="ays-pd-description" id="ays-pd-description-id">'. $pd_description .'</div>';
		$content[] = '<div id="' . $this->html_class_prefix . 'box_id' . '" class="' . $this->html_class_prefix . 'box' . " " . $custom_class .'">';
			
			$content[] = '<div id="' . $this->html_class_prefix . 'container_id' . '" class="' . $this->html_class_prefix . 'container ' . $menu_position_class . ' ' . $menu_position_mobile_class . '">';

				$content[] = $this->show_pd();

			$content[] = '</div>';

		$content[] = $this->get_styles();
		$content[] = $this->get_custom_css();
		$content[] = $this->get_encoded_options();
		$content[] = '</div>';

		$content = implode( '', $content );
		echo $content;
		return str_replace( array( "\r\n", "\n", "\r" ), '', ob_get_clean() );
	}

	public function ays_pd_add_word_from_front() {
		$pd = Personal_Dictionary_Data::get_pd_globsettings();
		$this->options = Personal_Dictionary_Data::get_pd_validated_data_from_array( $pd );
		echo $this->get_encoded_options();
		wp_enqueue_script( $this->plugin_name . '-shortcode-add-word-from-front',  plugin_dir_url( __FILE__ )  . 'js/personal-dictionary-public-add-word-from-front.js', array( 'jquery' ), $this->version, true );
		
		$group_name = $this->options[ $this->name_prefix . 'change_group_name' ];
		$translation_name = $this->options[ $this->name_prefix . 'change_translation_name' ];

		wp_localize_script( $this->plugin_name . '-shortcode-add-word-from-front', 'aysPersonalDictionaryFrontAjaxPublic', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
			'group' => __( $group_name, $this->plugin_name ),
			'translation' => __( $translation_name, $this->plugin_name ),
        ) );
	}

	public function show_pd(){

		$enable_full_screen_mode = $this->options[ $this->name_prefix . 'enable_full_screen_mode' ];
		$enable_full_screen_mode_mobile = $this->options[ $this->name_prefix . 'enable_full_screen_mode_mobile' ];
        $fullcsreen_mode = '';

        $fullcsreen_mode_html = '<div class="ays-pd-open-full-screen">
	                <a class="ays_pd_open_full_screen">
						<svg xmlns="http://www.w3.org/2000/svg" class="ays-pd-icon-main-color-fill" height="20" viewBox="0 0 24 24" width="20" class="open_full_screen">
	                        <path d="M0 0h24v24H0z" fill="none"/>
	                        <path d="M7 14H5v5h5v-2H7v-3zm-2-4h2V7h3V5H5v5zm12 7h-3v2h5v-5h-2v3zM14 5v2h3v3h2V5h-5z"/>
	                    </svg>
	                </a>
	            </div>';
		if ( ! wp_is_mobile() ) {
			if( $enable_full_screen_mode ){		
	            $fullcsreen_mode = $fullcsreen_mode_html;
	        }        	
        }
        if ( wp_is_mobile() ) {
	        if( $enable_full_screen_mode_mobile ){
	            $fullcsreen_mode = $fullcsreen_mode_html;
	        }        
        }

		$pd = Personal_Dictionary_Data::get_pd_globsettings();
		$this->options = Personal_Dictionary_Data::get_pd_validated_data_from_array( $pd );
		$group_name = $this->options[ $this->name_prefix . 'change_group_name' ];

		$texts = array(
			'groups' => __( $group_name, $this->plugin_name ),
			'games' => __( "Games", $this->plugin_name ),
			'flashcards' => __( "Flashcards", $this->plugin_name ),
			'import' => __( "Import", $this->plugin_name ),
		);

		$group_tab_url = esc_url_raw( add_query_arg( array("ays-pd-tab" => "groups")));
		$games_tab_url = esc_url_raw( add_query_arg( array("ays-pd-tab" => "games")));
		$flashcards_tab_url = esc_url_raw( add_query_arg( array("ays-pd-tab" => "flashcards")));
		$import_tab_url = esc_url_raw( add_query_arg( array("ays-pd-tab" => "import")));
		$content = array();

		$current_menu = (isset($_GET['ays-pd-tab']) && $_GET['ays-pd-tab'] != '' ) ? sanitize_text_field($_GET['ays-pd-tab']) : 'groups';

		$menu_name = isset( $texts[ $current_menu ] ) ? $texts[ $current_menu ] : $texts[ 'groups' ];
		$menu_name = '<h3>'. $menu_name .'</h3>';

		$active_tab_group = ($current_menu == 'groups') ? 'ays-pd-nav-tab-active' : '';
		$active_tab_games = ($current_menu == 'games') ? 'ays-pd-nav-tab-active' : '';
		$active_tab_flashcards = ($current_menu == 'flashcards') ? 'ays-pd-nav-tab-active' : '';
		$active_tab_import = ($current_menu == 'import') ? 'ays-pd-nav-tab-active' : '';

		$prevIcon = '<svg width="8" height="12" viewBox="0 0 8 12" xmlns="http://www.w3.org/2000/svg" class="ays-pd-icon-main-color-fill"><path d="M0.727757 5.99993C0.727685 5.92829 0.741769 5.85734 0.769198 5.79117C0.796627 5.72499 0.83686 5.66488 0.887588 5.6143L6.34231 0.159943C6.55545 -0.0531861 6.90059 -0.0531745 7.11358 0.159969C7.32657 0.373112 7.3267 0.718248 7.11355 0.93124L2.04448 5.99997L7.11321 11.069C7.32634 11.2822 7.32633 11.6273 7.11319 11.8403C6.90004 12.0533 6.55491 12.0534 6.34192 11.8403L0.887562 6.38557C0.836837 6.33498 0.796608 6.27487 0.769184 6.20869C0.74176 6.14251 0.727681 6.07156 0.727757 5.99993Z"/></svg>';
		$addWordIcon = '<svg xmlns="http://www.w3.org/2000/svg" class="ays-pd-icon-main-color-fill" width="12" height="12" viewBox="0 0 24 24" trtur><path d="M21.8906 9.89062H14.1094V2.10938C14.1094 0.944391 13.165 0 12 0C10.835 0 9.89062 0.944391 9.89062 2.10938V9.89062H2.10938C0.944391 9.89062 0 10.835 0 12C0 13.165 0.944391 14.1094 2.10938 14.1094H9.89062V21.8906C9.89062 23.0556 10.835 24 12 24C13.165 24 14.1094 23.0556 14.1094 21.8906V14.1094H21.8906C23.0556 14.1094 24 13.165 24 12C24 10.835 23.0556 9.89062 21.8906 9.89062Z"/></svg>';
		$sortIcon = '<svg xmlns="http://www.w3.org/2000/svg" class="ays-pd-icon-main-color-fill" width="14" height="14" viewBox="0 0 14 14" fill="none"><path d="M11.0753 1.52588e-05C10.9236 0.00187891 10.7786 0.0627681 10.671 0.169766L8.33767 2.50428C8.25612 2.58586 8.20058 2.68979 8.17808 2.80293C8.15558 2.91607 8.16714 3.03334 8.21128 3.13992C8.25542 3.24649 8.33017 3.33759 8.42607 3.40169C8.52198 3.46579 8.63474 3.50001 8.75009 3.50004H10.5001V10.5001H8.75009C8.63474 10.5001 8.52198 10.5343 8.42607 10.5984C8.33017 10.6625 8.25542 10.7536 8.21128 10.8602C8.16714 10.9668 8.15558 11.084 8.17808 11.1972C8.20058 11.3103 8.25612 11.4143 8.33767 11.4958L10.671 13.8292C10.7804 13.9385 10.9288 14 11.0834 14C11.2381 14 11.3865 13.9385 11.4959 13.8292L13.8292 11.4958C13.9108 11.4143 13.9663 11.3103 13.9888 11.1972C14.0113 11.084 13.9997 10.9668 13.9556 10.8602C13.9115 10.7536 13.8367 10.6625 13.7408 10.5984C13.6449 10.5343 13.5321 10.5001 13.4168 10.5001H11.6668V3.50004H13.4168C13.5321 3.50001 13.6449 3.46579 13.7408 3.40169C13.8367 3.33759 13.9115 3.24649 13.9556 3.13992C13.9997 3.03334 14.0113 2.91607 13.9888 2.80293C13.9663 2.68979 13.9108 2.58586 13.8292 2.50428L11.4959 0.169766C11.4407 0.114999 11.3751 0.0718205 11.3029 0.0427691C11.2308 0.0137177 11.1536 -0.00062027 11.0759 0.000598554L11.0753 1.52588e-05ZM0.59154 2.9167H6.40858C7.19725 2.92837 7.19725 1.73836 6.40858 1.75003H0.59154C0.513735 1.74885 0.436484 1.76327 0.364342 1.79244C0.292201 1.8216 0.226631 1.86492 0.1715 1.91983C0.116369 1.97474 0.0727938 2.04014 0.0433451 2.11217C0.0138964 2.18419 -0.000829583 2.26139 3.60643e-05 2.3392C0.00586943 2.66412 0.269538 2.92137 0.59154 2.9167ZM0.59154 7.5834H6.40858C7.19725 7.59506 7.19725 6.40506 6.40858 6.41672H0.59154C0.513735 6.41555 0.436484 6.42997 0.364342 6.45913C0.292201 6.4883 0.226631 6.53161 0.1715 6.58653C0.116369 6.64144 0.0727938 6.70684 0.0433451 6.77886C0.0138964 6.85089 -0.000829583 6.92808 3.60643e-05 7.00589C0.00586943 7.33081 0.269538 7.58806 0.59154 7.5834ZM0.59154 12.2501H6.40858C7.19725 12.2618 7.19725 11.0718 6.40858 11.0834H0.59154C0.513735 11.0822 0.436484 11.0967 0.364342 11.1258C0.292201 11.155 0.226631 11.1983 0.1715 11.2532C0.116369 11.3081 0.0727938 11.3735 0.0433451 11.4456C0.0138964 11.5176 -0.000829583 11.5948 3.60643e-05 11.6726C0.00586943 11.9975 0.269538 12.2548 0.59154 12.2501Z"/></svg>';
		$searchIcon ='<svg width="14" height="14" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><g stroke-width="0"></g><g stroke-linecap="round" stroke-linejoin="round"></g><g> <path d="M15.7955 15.8111L21 21M18 10.5C18 14.6421 14.6421 18 10.5 18C6.35786 18 3 14.6421 3 10.5C3 6.35786 6.35786 3 10.5 3C14.6421 3 18 6.35786 18 10.5Z" stroke="' . $this->options[ $this->name_prefix . 'main_color' ] . '" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"></path> </g></svg>';
		$showSearchIcon = ($current_menu == 'groups') ? '' : 'ays_display_none' ;

		$show_games = $this->options[ $this->name_prefix . 'show_games' ];

		$menu_layout_class = $this->html_class_prefix . 'menu-layout-' . $this->options[ $this->name_prefix . 'menu_layout' ];
		$menu_layout_mobile_class = $this->html_class_prefix . 'menu-layout-mobile-' . $this->options[ $this->name_prefix . 'menu_layout_mobile' ];

		$content[] = '<div class="' . $this->html_class_prefix . 'nav-menu ' . $menu_layout_class . ' ' . $menu_layout_mobile_class . '">';
			$content[] = '<div class="' . $this->html_class_prefix . 'nav-menu-item">';
				$content[] = '<a href="' . $group_tab_url . '"  class="' . $this->html_class_prefix . 'nav-tab ' . $active_tab_group . '">';
					$content[] = '<img src="'. PERSONAL_DICTIONARY_PUBLIC_URL .'/images/icons/group-white.svg" class=" ' . $this->html_class_prefix . 'nav-tab-image-white">';
					$content[] = '<img src="'. PERSONAL_DICTIONARY_PUBLIC_URL .'/images/icons/group-black.svg" class=" ' . $this->html_class_prefix . 'nav-tab-image-black">';
					$content[] = '<span>'. $texts['groups'] .'</span>';
				$content[] = '</a>';
			$content[] = '</div>';
			if ($show_games == 'on') {
				$content[] = '<div class="' . $this->html_class_prefix . 'nav-menu-item">';
					$content[] = '<a href="' . $games_tab_url . '"  class="' . $this->html_class_prefix . 'nav-tab ' . $active_tab_games . '">';
						$content[] = '<img src="'. PERSONAL_DICTIONARY_PUBLIC_URL .'/images/icons/game-white.svg" class=" ' . $this->html_class_prefix . 'nav-tab-image-white">';
						$content[] = '<img src="'. PERSONAL_DICTIONARY_PUBLIC_URL .'/images/icons/game-black.svg" class=" ' . $this->html_class_prefix . 'nav-tab-image-black">';
						$content[] = '<span>'. $texts['games'] .'</span>';
					$content[] = '</a>';
				$content[] = '</div>';
			}
			$content[] = '<div class="' . $this->html_class_prefix . 'nav-menu-item">';
				$content[] = '<a href="' . $flashcards_tab_url . '"  class="' . $this->html_class_prefix . 'nav-tab ' . $active_tab_flashcards . '">';
					$content[] = '<img src="'. PERSONAL_DICTIONARY_PUBLIC_URL .'/images/icons/flashcards-white.svg" class=" ' . $this->html_class_prefix . 'nav-tab-image-white">';
					$content[] = '<img src="'. PERSONAL_DICTIONARY_PUBLIC_URL .'/images/icons/flashcards-black.svg" class=" ' . $this->html_class_prefix . 'nav-tab-image-black">';
					$content[] = '<span>'. $texts['flashcards'] .'</span>';
				$content[] = '</a>';
			$content[] = '</div>';
			$content[] = '<div class="' . $this->html_class_prefix . 'nav-menu-item">';
				$content[] = '<a href="' . $import_tab_url . '"  class="' . $this->html_class_prefix . 'nav-tab ' . $active_tab_import . '">';
					$content[] = '<img src="'. PERSONAL_DICTIONARY_PUBLIC_URL .'/images/icons/import-white.svg" class=" ' . $this->html_class_prefix . 'nav-tab-image-white">';
					$content[] = '<img src="'. PERSONAL_DICTIONARY_PUBLIC_URL .'/images/icons/import-black.svg" class=" ' . $this->html_class_prefix . 'nav-tab-image-black">';
					$content[] = '<span>'. $texts['import'] .'</span>';
				$content[] = '</a>';
			$content[] = '</div>';
		$content[] = '</div>';

		$content[] = '<div class="' . $this->html_class_prefix . 'content' . ' ' . $this->html_class_prefix . 'content-' . $current_menu . '">';
			$content[] = '<div class="' . $this->html_class_prefix . 'content-div">';
				$content[] = '<div class="' . $this->html_class_prefix . 'header-wrap">';
					$content[] = '<div class="' . $this->html_class_prefix . 'header">';
						$content[] = '<div class="' . $this->html_class_prefix . 'header-main">';
							$content[] = '<div class="' . $this->html_class_prefix . 'header-previous-page-btn-box ' . $this->html_class_prefix . 'header-btn-box ays_display_none">';
								$content[] = '<div class="' . $this->html_class_prefix . 'header-previous-page-btn ' . $this->html_class_prefix . 'header-btn">';
									$content[] = $prevIcon;
								$content[] = '</div>';
							$content[] = '</div>';
							$content[] = $menu_name;
							$content[] = '<div class="' . $this->html_class_prefix . 'header-search-group-btn-box ' . $this->html_class_prefix . 'header-btn-box ' . $showSearchIcon . ' ">';
								$content[] = '<div class="' . $this->html_class_prefix . 'header-search-group-btn ' . $this->html_class_prefix . 'header-btn">';
									$content[] = $searchIcon;
								$content[] = '</div>';
							$content[] = '</div>';
							$content[] = '<div class="' . $this->html_class_prefix . 'header-search-word-btn-box ' . $this->html_class_prefix . 'header-btn-box ays_display_none">';
								$content[] = '<div class="' . $this->html_class_prefix . 'header-search-word-btn ' . $this->html_class_prefix . 'header-btn">';
									$content[] = $searchIcon;
								$content[] = '</div>';
							$content[] = '</div>';
							$content[] = '<div class="' . $this->html_class_prefix . 'header-add-word-btn-box ' . $this->html_class_prefix . 'header-btn-box ays_display_none">';
								$content[] = '<div class="' . $this->html_class_prefix . 'header-add-word-btn ' . $this->html_class_prefix . 'header-btn">';
									$content[] = $addWordIcon;
								$content[] = '</div>';
							$content[] = '</div>';
							$content[] = '<div class="' . $this->html_class_prefix . 'header-sort-btn-box ' . $this->html_class_prefix . 'header-btn-box ays_display_none">';
								$content[] = '<div class="' . $this->html_class_prefix . 'header-sort-btn ' . $this->html_class_prefix . 'header-btn">';
									$content[] = $sortIcon;
								$content[] = '</div>';
								$content[] = '<div class="' . $this->html_class_prefix . 'sort-words-in-group-popup"></div>';
							$content[] = '</div>';
						$content[] = '</div>';
						$content[] = '<div class="' . $this->html_class_prefix . 'header-full-screen">';
							$content[] = $fullcsreen_mode;
						$content[] = '</div>';
					$content[] = '</div>';
				$content[] = '</div>';
				switch ($current_menu) {
					case 'groups':
						$content[] = $this->groups_tab();
						break;
					case 'games':
						$content[] = $this->games_tab();
						break;
					case 'flashcards':
						$content[] = $this->flashcards_tab();
						break;
					case 'import':
						$content[] = $this->import_tab();
						break;
					default:
						$content[] = $this->groups_tab();
						break;
				}
			$content[] = '</div>';
		$content[] = '</div>';
		$content = implode( '', $content );
		return $content;
	}

	public function groups_tab() {
		$content = array();
		$content[] = '<div class="' . $this->html_class_prefix . 'search-groups-container '.  $this->html_class_prefix . 'group-tab-add-layer">';
		$content[] = '</div>';
		$content[] = '<div class="' . $this->html_class_prefix . 'search-word-container">';
		$content[] = '</div>';
		$content[] = '<div class="' . $this->html_class_prefix . 'save-groups-block '.  $this->html_class_prefix . 'group-tab-add-layer" data-function="">';
		$content[] = '</div>';
		$content[] = '<div class="' . $this->html_class_prefix . 'group-tab-words '. $this->html_class_prefix . 'group-tab-add-layer" data-function="ays_groups_pd">';
		$content[] = '</div>';
		$content[] = '<div class="' . $this->html_class_prefix . 'group-tab '. $this->html_class_prefix . 'tab-content" data-function="ays_groups_pd"></div>';
		$content[] =  '<div class="ays-pd-preloader">
                <img class="loader" src="'. PERSONAL_DICTIONARY_PUBLIC_URL .'/images/loaders/3-1.svg">
            </div>';
		
		$content = implode( '',$content );
		return $content;
	}
	
	public function games_tab() {
		$content = array();
		$findTheWordIcon = '<svg xmlns="http://www.w3.org/2000/svg" width="33" height="24" viewBox="0 0 33 24" class="ays-pd-icon-main-color-fill" >
		<path d="M31.6412 18.8334L24.4847 11.6752C25.8275 8.62665 25.2529 4.93029 22.76 2.43674C19.512 -0.812029 14.2546 -0.812467 11.0062 2.43674C7.76577 5.67789 7.76577 10.9516 11.0062 14.1928C13.5812 16.7683 17.3548 17.2587 20.3674 15.866L27.4874 22.9877C28.6353 24.1359 30.4935 24.136 31.6415 22.9874C32.7863 21.8421 32.7862 19.9787 31.6412 18.8334ZM12.3322 12.867C9.82261 10.3569 9.82261 6.27267 12.3322 3.76249C14.8473 1.24675 18.9184 1.24631 21.434 3.76249C23.9435 6.27267 23.9435 10.3569 21.434 12.867C18.9188 15.3828 14.8477 15.3832 12.3322 12.867ZM30.3153 21.6619C29.9003 22.0771 29.2285 22.0773 28.8133 21.6619L22.0109 14.8579C22.5699 14.4184 23.0707 13.9095 23.5011 13.3434L30.3152 20.1591C30.7294 20.5734 30.7294 21.2475 30.3153 21.6619ZM1.43754 11.2343H5.43535C5.95312 11.2343 6.37289 10.8146 6.37289 10.2968C6.37289 9.77901 5.95312 9.35925 5.43535 9.35925H1.43754C0.919769 9.35925 0.5 9.77901 0.5 10.2968C0.5 10.8146 0.919769 11.2343 1.43754 11.2343ZM1.43754 15.2345H6.93454C7.45232 15.2345 7.87208 14.8147 7.87208 14.297C7.87208 13.7792 7.45232 13.3594 6.93454 13.3594H1.43754C0.919769 13.3594 0.5 13.7792 0.5 14.297C0.5 14.8147 0.919769 15.2345 1.43754 15.2345ZM1.43754 19.2347H10.9324C11.4502 19.2347 11.87 18.8149 11.87 18.2972C11.87 17.7794 11.4502 17.3596 10.9324 17.3596H1.43754C0.919769 17.3596 0.5 17.7794 0.5 18.2972C0.5 18.8149 0.919769 19.2347 1.43754 19.2347ZM21.9265 21.3598H1.43754C0.919769 21.3598 0.5 21.7796 0.5 22.2973C0.5 22.8151 0.919769 23.2349 1.43754 23.2349H21.9265C22.4443 23.2349 22.8641 22.8151 22.8641 22.2973C22.8641 21.7796 22.4443 21.3598 21.9265 21.3598Z"/>
		</svg>';
		$findTheTranslationIcon = '<svg xmlns="http://www.w3.org/2000/svg" width="29" height="28" viewBox="0 0 29 28" fill="none">
		<path d="M12.6381 16.7537L10.7631 7.37862C10.7206 7.16611 10.6058 6.97489 10.4382 6.83749C10.2706 6.70008 10.0606 6.625 9.84389 6.625H7.96889C7.75217 6.625 7.54214 6.70008 7.37455 6.83749C7.20695 6.97489 7.09214 7.16611 7.04964 7.37862L5.17464 16.7536C5.07308 17.2614 5.40239 17.7553 5.91008 17.8568C6.41777 17.9583 6.9117 17.629 7.0132 17.1213L7.61245 14.125H10.2003L10.7996 17.1214C10.9011 17.6293 11.3953 17.9584 11.9027 17.8568C12.4104 17.7553 12.7397 17.2614 12.6381 16.7537ZM7.98745 12.25L8.73745 8.5H9.07533L9.82533 12.25H7.98745ZM27.6564 14.125H24.8439V13.1875C24.8439 12.6698 24.4241 12.25 23.9064 12.25C23.3886 12.25 22.9689 12.6698 22.9689 13.1875V14.125H20.1564C19.6386 14.125 19.2189 14.5448 19.2189 15.0625C19.2189 15.5802 19.6386 16 20.1564 16H20.3833C20.9175 17.7257 21.7217 19.0504 22.6067 20.1119C21.8864 20.7708 21.1575 21.3111 20.5083 21.8304C20.104 22.1539 20.0384 22.7438 20.3619 23.1481C20.6855 23.5526 21.2755 23.6178 21.6796 23.2945C22.3325 22.7722 23.115 22.1917 23.9064 21.4655C24.6983 22.1923 25.4823 22.7738 26.1332 23.2945C26.5375 23.618 27.1275 23.5524 27.4509 23.1481C27.7743 22.7438 27.7088 22.1538 27.3045 21.8304C26.657 21.3122 25.9271 20.7714 25.2061 20.1119C26.0911 19.0504 26.8953 17.7257 27.4295 16H27.6564C28.1741 16 28.5939 15.5802 28.5939 15.0625C28.5939 14.5448 28.1741 14.125 27.6564 14.125ZM23.9064 18.7341C23.308 17.9828 22.7696 17.0861 22.3637 15.9937H25.449C25.0432 17.0861 24.5048 17.9828 23.9064 18.7341Z" class="ays-pd-icon-main-color-fill"/>
		<path d="M25.9453 4.97656H14.1109L13.7592 2.15567C13.6056 0.926734 12.5558 0 11.3173 0H2.86719C1.51023 0 0.40625 1.10398 0.40625 2.46094V20.5625C0.40625 21.9195 1.51023 23.0234 2.86719 23.0234H9.78423L10.1314 25.8443C10.2847 27.0708 11.3345 28 12.5733 28H25.9453C27.3023 28 28.4062 26.896 28.4062 25.5391V7.4375C28.4062 6.08054 27.3023 4.97656 25.9453 4.97656ZM2.86719 21.3828C2.41487 21.3828 2.04688 21.0148 2.04688 20.5625V2.46094C2.04688 2.00862 2.41487 1.64062 2.86719 1.64062H11.3173C11.7302 1.64062 12.0801 1.9495 12.1312 2.35889L14.5027 21.3828H2.86719ZM11.7115 25.252L11.4372 23.0234H13.6381L11.7115 25.252ZM26.7656 25.5391C26.7656 25.9914 26.3976 26.3594 25.9453 26.3594H12.9228L16.0457 22.7471C16.1244 22.6584 16.1829 22.5537 16.2172 22.4401C16.2514 22.3266 16.2606 22.207 16.2441 22.0895L14.3154 6.61719H25.9453C26.3976 6.61719 26.7656 6.98518 26.7656 7.4375V25.5391Z" class="ays-pd-icon-main-color-fill"/>
		</svg>';
		$matchingWordsIcon = '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 33 33" fill="none">
		<g clip-path="url(#clip0_51_335)">
		<path d="M30.0844 0.5H19.6876C19.1698 0.5 18.7501 0.91975 18.7501 1.4375V5.74738C18.7501 5.92838 18.8025 6.10552 18.901 6.25739C18.9994 6.40926 19.1398 6.52938 19.305 6.60323C19.4703 6.67709 19.6534 6.70152 19.8322 6.67359C20.0111 6.64565 20.178 6.56655 20.3129 6.44581C20.5607 6.22323 20.8858 6.10657 21.2186 6.12081C21.8811 6.147 22.4196 6.6865 22.4446 7.349C22.4579 7.70119 22.3313 8.03431 22.0879 8.28706C21.8443 8.54006 21.5168 8.67937 21.1656 8.67937C20.8509 8.67954 20.5472 8.56342 20.3128 8.35331C20.1779 8.23258 20.011 8.15347 19.8322 8.12553C19.6533 8.0976 19.4702 8.12204 19.305 8.1959C19.1397 8.26977 18.9994 8.38989 18.9009 8.54177C18.8024 8.69366 18.75 8.8708 18.75 9.05181V13.3125C18.75 13.8302 19.1697 14.25 19.6875 14.25H22.4916C22.4536 14.4704 22.439 14.6942 22.4481 14.9177C22.5127 16.5524 23.8441 17.8815 25.479 17.9433C25.5203 17.9448 25.5613 17.9456 25.6024 17.9456C26.4227 17.9456 27.1949 17.6349 27.7884 17.0635C28.4119 16.4632 28.7553 15.656 28.7553 14.7907C28.7553 14.6082 28.7395 14.4274 28.7087 14.2501H31.5626C32.0803 14.2501 32.5001 13.8303 32.5001 13.3126V2.91569C32.5001 1.58369 31.4164 0.5 30.0844 0.5ZM30.6251 12.375H27.2527C27.0716 12.3749 26.8945 12.4273 26.7425 12.5258C26.5906 12.6243 26.4705 12.7646 26.3966 12.9299C26.3227 13.0952 26.2983 13.2784 26.3263 13.4572C26.3543 13.6361 26.4334 13.803 26.5543 13.9379C26.7643 14.1723 26.8805 14.476 26.8803 14.7907C26.8803 15.1417 26.741 15.4693 26.488 15.7129C26.2353 15.9563 25.9034 16.0824 25.5499 16.0696C24.8874 16.0446 24.3479 15.506 24.3218 14.8436C24.3085 14.5084 24.4239 14.1867 24.6467 13.9379C24.7674 13.803 24.8465 13.636 24.8744 13.4572C24.9023 13.2784 24.8778 13.0953 24.804 12.93C24.7301 12.7648 24.61 12.6244 24.4581 12.5259C24.3063 12.4275 24.1291 12.375 23.9481 12.375H20.6251V10.5079C20.8024 10.5387 20.9831 10.5544 21.1654 10.5544H21.1658C22.0311 10.5544 22.8383 10.2111 23.4386 9.58762C24.0387 8.96444 24.3511 8.14425 24.3183 7.27825C24.2566 5.64331 22.9275 4.31194 21.2926 4.24731C21.0692 4.23827 20.8454 4.25289 20.6251 4.29094V2.375H30.0844C30.3826 2.375 30.6251 2.61756 30.6251 2.91569V12.375Z" class="ays-pd-icon-main-color-fill"/>
		<path d="M25.1875 18.75H20.8776C20.6966 18.7499 20.5194 18.8023 20.3675 18.9008C20.2156 18.9993 20.0954 19.1396 20.0215 19.3049C19.9477 19.4702 19.9233 19.6534 19.9512 19.8322C19.9792 20.0111 20.0584 20.178 20.1792 20.3129C20.3893 20.5473 20.5054 20.851 20.5053 21.1658C20.5053 21.5168 20.3659 21.8443 20.1129 22.0879C19.8602 22.3313 19.5287 22.4571 19.1749 22.4446C18.5123 22.4196 17.9729 21.881 17.9467 21.2186C17.9334 20.8834 18.0489 20.5618 18.2716 20.3129C18.3923 20.178 18.4714 20.011 18.4993 19.8322C18.5272 19.6534 18.5028 19.4703 18.4289 19.305C18.3551 19.1398 18.2349 18.9994 18.0831 18.9009C17.9312 18.8025 17.7541 18.75 17.5731 18.75H14.25V16.8834C14.4681 16.921 14.6917 16.936 14.9176 16.9269C16.5524 16.8623 17.8815 15.5309 17.9433 13.896C17.976 13.03 17.6636 12.2098 17.0635 11.5866C16.4632 10.9631 15.656 10.6198 14.7907 10.6198C14.6082 10.6198 14.4274 10.6356 14.2501 10.6664V7.8125C14.2501 7.29475 13.8303 6.875 13.3126 6.875H2.91569C1.58369 6.875 0.5 7.95869 0.5 9.29069V30.0842C0.5 31.4163 1.58369 32.5 2.91569 32.5H23.7092C25.0413 32.5 26.125 31.4163 26.125 30.0843V19.6875C26.125 19.1698 25.7052 18.75 25.1875 18.75ZM2.375 9.29069C2.375 8.9925 2.61756 8.75 2.91569 8.75H12.375V12.1224C12.375 12.4922 12.5924 12.8274 12.93 12.9782C13.0953 13.0521 13.2784 13.0766 13.4572 13.0487C13.6361 13.0207 13.803 12.9416 13.9379 12.8208C14.1723 12.6107 14.476 12.4946 14.7907 12.4948C15.1417 12.4948 15.4693 12.6341 15.7129 12.8871C15.9563 13.1398 16.0829 13.4729 16.0696 13.8251C16.0446 14.4876 15.506 15.0271 14.8436 15.0533C14.5085 15.0665 14.1867 14.9511 13.9379 14.7284C13.803 14.6077 13.6361 14.5286 13.4572 14.5007C13.2784 14.4727 13.0953 14.4972 12.9301 14.571C12.7648 14.6449 12.6245 14.765 12.526 14.9168C12.4275 15.0687 12.3751 15.2458 12.3751 15.4268V18.75H10.508C10.5388 18.5726 10.5545 18.3918 10.5545 18.2093C10.5545 17.344 10.2112 16.5368 9.58769 15.9366C8.9645 15.3364 8.14381 15.0229 7.27831 15.0568C5.64338 15.1186 4.312 16.4476 4.24738 18.0824C4.23845 18.3059 4.25305 18.5297 4.29094 18.7501H2.375V9.29069ZM2.375 30.0843V20.625H5.74738C5.92841 20.625 6.10559 20.5727 6.2575 20.4742C6.40941 20.3757 6.52955 20.2353 6.60341 20.07C6.67727 19.9048 6.70169 19.7216 6.67371 19.5428C6.64573 19.3639 6.56656 19.197 6.44575 19.0621C6.223 18.8133 6.10756 18.4916 6.12081 18.1565C6.147 17.494 6.6865 16.9554 7.349 16.9304C7.70038 16.9184 8.03425 17.0437 8.28706 17.2872C8.54006 17.5308 8.67937 17.8582 8.67937 18.2093C8.67952 18.5241 8.56341 18.8278 8.35331 19.0622C8.10675 19.3377 8.04506 19.7325 8.19594 20.0701C8.34681 20.4076 8.68219 20.625 9.05194 20.625H12.375V22.4916C12.1546 22.4537 11.9308 22.4391 11.7074 22.4481C10.0726 22.5127 8.7435 23.8441 8.68175 25.479C8.649 26.345 8.96144 27.1652 9.5615 27.7884C10.1618 28.4119 10.969 28.7552 11.8343 28.7552C12.0168 28.7552 12.1976 28.7394 12.3749 28.7086V30.625H2.91569C2.61756 30.625 2.375 30.3824 2.375 30.0843ZM24.25 30.0843C24.25 30.3825 24.0074 30.625 23.7093 30.625H14.25V27.2526C14.2501 27.0716 14.1977 26.8944 14.0992 26.7425C14.0007 26.5906 13.8604 26.4704 13.6951 26.3965C13.5298 26.3227 13.3466 26.2983 13.1678 26.3262C12.9889 26.3542 12.822 26.4334 12.6871 26.5542C12.4527 26.7643 12.149 26.8804 11.8343 26.8803C11.4833 26.8803 11.1557 26.7409 10.9121 26.4879C10.6687 26.2352 10.5421 25.9021 10.5554 25.5499C10.5804 24.8874 11.119 24.3479 11.7814 24.3217C12.1142 24.3077 12.4392 24.4243 12.6871 24.6466C12.822 24.7674 12.9889 24.8465 13.1678 24.8744C13.3466 24.9023 13.5297 24.8779 13.695 24.804C13.8602 24.7302 14.0006 24.6101 14.099 24.4582C14.1975 24.3063 14.2499 24.1292 14.2499 23.9482V20.625H16.1166C16.0786 20.8454 16.064 21.0692 16.073 21.2926C16.1376 22.9274 17.469 24.2565 19.1039 24.3183C19.9696 24.3519 20.7901 24.0386 21.4133 23.4385C22.0368 22.8382 22.3801 22.031 22.3801 21.1658C22.3801 20.9833 22.3644 20.8024 22.3336 20.6251H24.25V30.0843Z" class="ays-pd-icon-main-color-fill"/>
		</g>
		<defs>
		<clipPath id="clip0_51_335">
		<rect width="32" height="32" fill="white" transform="translate(0.5 0.5)"/>
		</clipPath>
		</defs>
		</svg>';
		$writeTheTranslationIcon = '<svg  width="32" height="32" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><g stroke-width="0"></g><g stroke-linecap="round" stroke-linejoin="round"></g><g><path d="M1 22C1 21.4477 1.44772 21 2 21H22C22.5523 21 23 21.4477 23 22C23 22.5523 22.5523 23 22 23H2C1.44772 23 1 22.5523 1 22Z" fill="' . $this->options[ $this->name_prefix . 'main_color' ] . '"></path> <path fill-rule="evenodd" clip-rule="evenodd" d="M18.3056 1.87868C17.1341 0.707107 15.2346 0.707107 14.063 1.87868L3.38904 12.5526C2.9856 12.9561 2.70557 13.4662 2.5818 14.0232L2.04903 16.4206C1.73147 17.8496 3.00627 19.1244 4.43526 18.8069L6.83272 18.2741C7.38969 18.1503 7.89981 17.8703 8.30325 17.4669L18.9772 6.79289C20.1488 5.62132 20.1488 3.72183 18.9772 2.55025L18.3056 1.87868ZM15.4772 3.29289C15.8677 2.90237 16.5009 2.90237 16.8914 3.29289L17.563 3.96447C17.9535 4.35499 17.9535 4.98816 17.563 5.37868L15.6414 7.30026L13.5556 5.21448L15.4772 3.29289ZM12.1414 6.62869L4.80325 13.9669C4.66877 14.1013 4.57543 14.2714 4.53417 14.457L4.0014 16.8545L6.39886 16.3217C6.58452 16.2805 6.75456 16.1871 6.88904 16.0526L14.2272 8.71448L12.1414 6.62869Z" class="ays-pd-icon-main-color-fill"/></path> </g></svg>';
	
		$pd = Personal_Dictionary_Data::get_pd_globsettings();
		$this->options = Personal_Dictionary_Data::get_pd_validated_data_from_array( $pd );
		$group_name = $this->options[ $this->name_prefix . 'change_group_name' ];
		$word_name = $this->options[ $this->name_prefix . 'change_word_name' ];
		$translation_name = $this->options[ $this->name_prefix . 'change_translation_name' ];

		$show_games = $this->options[ $this->name_prefix . 'show_games' ];
		$show_game_find_the_word = $this->options[ $this->name_prefix . 'show_game_find_the_word' ];
		$show_game_find_the_translation = $this->options[ $this->name_prefix . 'show_game_find_the_translation' ];
		$show_game_matching_words = $this->options[ $this->name_prefix . 'show_game_matching_words' ];
		$show_game_write_the_translation = $this->options[ $this->name_prefix . 'show_game_write_the_translation' ];
		$games_animation_effect = $this->options[ $this->name_prefix . 'games_animation_effect' ];
		$games_animation_effect_mobile = $this->options[ $this->name_prefix . 'games_animation_effect_mobile' ];
		
		$animation_effect = 'none';
		if ( ! wp_is_mobile() ) {
	        if( $games_animation_effect ){
	            $animation_effect = $games_animation_effect;
	        }        	
        }
        if ( wp_is_mobile() ) {
	        if( $games_animation_effect_mobile ){
	            $animation_effect = $games_animation_effect_mobile;
	        }      
		}
		if ($show_games == 'on') {
			$content[] = '<div class="' . $this->html_class_prefix . 'games-tab '. $this->html_class_prefix . 'tab-content'.'" data-animation-effect="' . $animation_effect . '" data-function="ays_games_pd">';
				$content[] = '<div class="' . $this->html_class_prefix . 'games-choosing-type">';

					if($show_game_find_the_word == 'on') {
						$content[] = '<div class="' . $this->html_class_prefix . 'games-choosing-type-each">';
							$content[] = '<label>';
								$content[] = '<input class="' . $this->html_class_prefix . 'game-type-rad'.'" id="' . $this->html_class_prefix . 'games-type-find-word'. '" type="radio" name="ays-pd[game_type]" value="find_word">';
								$content[] = '<div class="' . $this->html_class_prefix . 'game-type-item">';
									$content[] = '<div class="' . $this->html_class_prefix . 'game-type-item-image">' . $findTheWordIcon . '</div>';
									$content[] = '<div class="' . $this->html_class_prefix . 'game-type-item-title">'. __("Find the " . strtolower($word_name), $this->plugin_name ) . '</div>';
								$content[] = '</div>';
							$content[] = '</label>';
						$content[] = '</div>';
					}

					if($show_game_find_the_translation == 'on') {
						$content[] = '<div class="' . $this->html_class_prefix . 'games-choosing-type-each">';
							$content[] = '<label>';
								$content[] = '<input class="' . $this->html_class_prefix . 'game-type-rad'.'" id="' . $this->html_class_prefix . 'games-type-find-translation'. '" type="radio" name="ays-pd[game_type]" value="find_translation">';
								$content[] = '<div class="' . $this->html_class_prefix . 'game-type-item">';
									$content[] = '<div class="' . $this->html_class_prefix . 'game-type-item-image">' . $findTheTranslationIcon . '</div>';
									$content[] = '<div class="' . $this->html_class_prefix . 'game-type-item-title">'. __("Find the ". strtolower($translation_name), $this->plugin_name ) . '</div>';
								$content[] = '</div>';
							$content[] = '</label>';
						$content[] = '</div>';
					}

					if($show_game_matching_words == 'on') {
						$content[] = '<div class="' . $this->html_class_prefix . 'games-choosing-type-each">';
							$content[] = '<label>';
								$content[] = '<input class="' . $this->html_class_prefix . 'game-type-rad'.'" id="' . $this->html_class_prefix . 'games-type-matching_words'. '" type="radio" name="ays-pd[game_type]" value="matching_words">';
								$content[] = '<div class="' . $this->html_class_prefix . 'game-type-item">';
									$content[] = '<div class="' . $this->html_class_prefix . 'game-type-item-image">' . $matchingWordsIcon . '</div>';
									$content[] = '<div class="' . $this->html_class_prefix . 'game-type-item-title">'. __("Matching " . strtolower($word_name), $this->plugin_name ) . '</div>';
								$content[] = '</div>';
							$content[] = '</label>';
						$content[] = '</div>';
					}

					if($show_game_write_the_translation == 'on') {
						$content[] = '<div class="' . $this->html_class_prefix . 'games-choosing-type-each">';
							$content[] = '<label>';
								$content[] = '<input class="' . $this->html_class_prefix . 'game-type-rad'.'" id="' . $this->html_class_prefix . 'games-type-write-translation'. '" type="radio" name="ays-pd[game_type]" value="write_translation">';
								$content[] = '<div class="' . $this->html_class_prefix . 'game-type-item">';
									$content[] = '<div class="' . $this->html_class_prefix . 'game-type-item-image">' . $writeTheTranslationIcon . '</div>';
									$content[] = '<div class="' . $this->html_class_prefix . 'game-type-item-title">'. __("Write the " . strtolower($translation_name), $this->plugin_name ) . '</div>';
								$content[] = '</div>';
							$content[] = '</label>';
						$content[] = '</div>';
					}

				$content[] = '</div>';
	
				$content[] = '<div class="' . $this->html_class_prefix . 'games-type-content" style="display:none">';
					$content[] = '<div class="' . $this->html_class_prefix . 'games-type-content-settings"></div>';
					$content[] = '<form method="POST" class="' . $this->html_class_prefix . 'games-type-content-game"></form>';
				$content[] = '</div>';

			$content[] = '</div>';

			$content[] = '<div class="ays-pd-preloader">
				<img class="loader" src="'. PERSONAL_DICTIONARY_PUBLIC_URL .'/images/loaders/3-1.svg">
			</div>';
		}

		$content = implode( '', $content );
		return $content;
	}

	public function flashcards_tab() {
		$pd = Personal_Dictionary_Data::get_pd_globsettings();
		$this->options = Personal_Dictionary_Data::get_pd_validated_data_from_array( $pd );
		$group_name = $this->options[ $this->name_prefix . 'change_group_name' ];
		$word_name = $this->options[ $this->name_prefix . 'change_word_name' ];

		$content = array();

		$content[] = '<div class="' . $this->html_class_prefix . 'flashcards-tab '. $this->html_class_prefix . 'tab-content" data-function="ays_pd_get_groups">';
			$content[] = '<div class="' . $this->html_class_prefix . 'flashcards-content">';
				$content[] = '<div class="' . $this->html_class_prefix . 'flashcards-settings-content">';
					$content[] = '<div class="' . $this->html_class_prefix . 'flashcards-select-groups">';
						$content[] = '<select class="' . $this->html_class_prefix . 'flashcards-settings-item ' . $this->html_class_prefix . 'flashcards-group-item" multiple></select>';
						$content[] = '<div class="' . $this->html_class_prefix . 'flashcards-word-count-explanation">';
							$content[] = '<p class="' . $this->html_class_prefix . 'flashcards-word-count-explanation-paragraph">' . __('Select maximum '. strtolower($word_name) .' count:', $this->plugin_name) . '</p>';
						$content[] = '</div>';
						$content[] = '<div class="' . $this->html_class_prefix . 'flashcards-settings-item ' . $this->html_class_prefix . 'flashcards-check-words-count" >';	
							$content[] = '<label class="' . $this->html_class_prefix . 'form-check-label ' . $this->html_class_prefix . 'form-check-label-active">';
								$content[] = '<input class="' . $this->html_class_prefix . 'flashcards-words-count" type="radio" name="words-count-rad" value="10" checked>';
								$content[] = '<span>10</span>';
							$content[] = '</label>';
							$content[] = '<label class="' . $this->html_class_prefix . 'form-check-label">';
								$content[] = '<input class="' . $this->html_class_prefix . 'flashcards-words-count" type="radio" name="words-count-rad" value="30">';
								$content[] = '<span>30</span>';
							$content[] = '</label>';
							$content[] = '<label class="' . $this->html_class_prefix . 'form-check-label">';
								$content[] = '<input class="' . $this->html_class_prefix . 'flashcards-words-count" type="radio" name="words-count-rad" value="50">';
								$content[] = '<span>50</span>';
							$content[] = '</label>';
							$content[] = '<label class="' . $this->html_class_prefix . 'form-check-label">';
								$content[] = '<input class="' . $this->html_class_prefix . 'flashcards-words-count" type="radio" name="words-count-rad" value="100">';
								$content[] = '<span>100</span>';
							$content[] = '</label>';
							$content[] = '<label class="' . $this->html_class_prefix . 'form-check-label">';
								$content[] = '<input class="' . $this->html_class_prefix . 'flashcards-words-count" type="radio" name="words-count-rad" value="All">';
								$content[] = '<span>' . __('All', $this->plugin_name) . '</span>';
							$content[] = '</label>';
						$content[] = '</div>';
						$content[] = '<div class="' . $this->html_class_prefix . 'flashcards-settings-item ' . $this->html_class_prefix . 'flashcards-start-btn-block" >';	
							$content[] = '<input type="button" value="Start" id="' . $this->html_class_prefix . 'flashcards-start-button">';
							$content[] = '<p id="' . $this->html_class_prefix . 'flashcards-no-group-message">' . __('Please select at least one '. strtolower($group_name) . '.', $this->plugin_name) . '</p>';
							$content[] = '<p id="' . $this->html_class_prefix . 'flashcards-no-words-message">' . __('The selected '. strtolower($group_name) .'(s) have no ' . strtolower($word_name) . '.', $this->plugin_name) . '</p>';
						$content[] = '</div>';
					$content[] = '</div>';
				$content[] = '</div>';
				$content[] = '<div class="' . $this->html_class_prefix . 'flashcards-content-main" style="display:none">';
					$content[] = '<div class="' . $this->html_class_prefix . 'flashcards-main-container">';
						$content[] = '<div class="' . $this->html_class_prefix . 'flashcards-main-inner-container">';
							$content[] = '<div class="' . $this->html_class_prefix . 'flashcard-box"></div>';
							$content[] = '<div class="' . $this->html_class_prefix . 'flashcard-data-box">';
								$content[] = '<div class="' . $this->html_class_prefix . 'flashcard-data-buttons-box">';
									$content[] = '<div class="' . $this->html_class_prefix . 'flashcard-data-box-prev-button-box" data-prev-word="">';
										$content[] = '<div class="' . $this->html_class_prefix . 'flashcard-data-box-prev-button ' . $this->html_class_prefix . 'flashcard-data-box-prev-button-disabled">';
											$content[] = '<img src="'.PERSONAL_DICTIONARY_PUBLIC_URL.'/images/icons/back.png" class="' . $this->html_class_prefix . 'flashcard-data-box-prev-button-icon">';
										$content[] = '</div>';
									$content[] = '</div>';
									$content[] = '<div class="' . $this->html_class_prefix . 'flashcard-data-box-know-dknow">';
										$content[] = '<div class="' . $this->html_class_prefix . 'flashcard-data-box-dknow ' . $this->html_class_prefix . 'flashcard-data-box-check-answer" data-type="dknow">';
											$content[] = '<img src="'.PERSONAL_DICTIONARY_PUBLIC_URL.'/images/icons/wrong.png" class="' . $this->html_class_prefix . 'flashcard-data-box-dknow-button">';
										$content[] = '</div>';
										$content[] = '<div class="' . $this->html_class_prefix . 'flashcard-data-words-count" title="'.__($word_name . ' count' , $this->plugin_name).'"> <span class="' . $this->html_class_prefix . 'flashcard-data-words-count-text">1</span> / <span class="' . $this->html_class_prefix . 'flashcard-data-all-questions-count"></span></div>';
										$content[] = '<div class="' . $this->html_class_prefix . 'flashcard-data-box-know ' . $this->html_class_prefix . 'flashcard-data-box-check-answer" data-type="know">';
											$content[] = '<img src="'.PERSONAL_DICTIONARY_PUBLIC_URL.'/images/icons/right.png" class="' . $this->html_class_prefix . 'flashcard-data-box-know-button">';
										$content[] = '</div>';
									$content[] = '</div>';
									$content[] = '<div class="' . $this->html_class_prefix . 'flashcard-data-box-settings" style="width: 20px"></div>';
								$content[] = '</div>';
								$content[] = '<div class="' . $this->html_class_prefix . 'flashcard-data-progress-bar-box">';
									$content[] = '<div class="' . $this->html_class_prefix . 'flashcard-data-progress-bar-outer" >';
										$content[] = '<div class="' . $this->html_class_prefix . 'flashcard-data-progress-bar-inner" data-value=""></div>';
									$content[] = '</div>';
								$content[] = '</div>';
							$content[] = '</div>';
						$content[] = '</div>';
					$content[] = '</div>';
				$content[] = '</div>';
			$content[] = '</div>';
		$content[] = '</div>';

		$content[] = '<div class="ays-pd-preloader">
				<img class="loader" src="'. PERSONAL_DICTIONARY_PUBLIC_URL .'/images/loaders/3-1.svg">
			</div>';

		$content = implode( '', $content );
		return $content;
	}

	public function import_tab(){
		global $wpdb;
		$content = array();
		if(isset( $_POST['ays-pd-import-save-btn'])){
			if( isset( $_FILES['ays-pd-import-file'] ) ){
				$stats = $this->ays_import_pd($_FILES['ays-pd-import-file']);

				$url = esc_url_raw( remove_query_arg('ays-pd-tab') );
				wp_redirect( $url );
			}
		}
		$user_id = get_current_user_id();
		$categories_table = esc_sql($wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'categories');
		$sql  = "SELECT * FROM ".$categories_table ." WHERE user_id = " . $user_id ."  ORDER BY id DESC";

		$uploadIcon = '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="24" viewBox="0 0 32 24" fill="none">
		<path d="M25.0037 23.284H18.1381V16.3465H20.407C20.9824 16.3465 21.3224 15.6927 20.9824 15.2219L16.5689 9.11483C16.2877 8.72252 15.7058 8.72252 15.4246 9.11483L11.011 15.2219C10.671 15.6927 11.0045 16.3465 11.5864 16.3465H13.8553V23.284H6.16592C2.73314 23.0944 0 19.8839 0 16.4054C0 14.0057 1.30119 11.9134 3.23008 10.7822C3.05354 10.3049 2.96199 9.79485 2.96199 9.25868C2.96199 6.8067 4.9432 4.8255 7.39518 4.8255C7.92481 4.8255 8.43482 4.91704 8.91214 5.09358C10.331 2.08582 13.3911 0 16.9481 0C21.5513 0.00653862 25.3437 3.53085 25.7752 8.02289C29.3126 8.63098 32 11.9068 32 15.6142C32 19.5766 28.9138 23.0094 25.0037 23.284Z" fill="' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color' ] , 0.2 ) . '"/>
		</svg>';
		$exportIcon = '<svg xmlns="http://www.w3.org/2000/svg" width="13" height="14" viewBox="0 0 13 14" fill="none">
		<path d="M4.35264 4.19337L5.84951 2.6965V8.94431C5.84951 9.33479 6.10983 9.59512 6.50032 9.59512C6.89081 9.59512 7.15114 9.33479 7.15114 8.94431V2.6965L8.64801 4.19337C8.90833 4.45369 9.29882 4.45369 9.55915 4.19337C9.81947 3.93304 9.81947 3.54255 9.55915 3.28223L6.95589 0.678974C6.89081 0.613892 6.82573 0.548811 6.76065 0.548811C6.63048 0.48373 6.43524 0.48373 6.24 0.548811C6.17492 0.548811 6.10983 0.613892 6.04475 0.678974L3.4415 3.28223C3.18117 3.54255 3.18117 3.93304 3.4415 4.19337C3.70182 4.45369 4.09231 4.45369 4.35264 4.19337ZM12.3576 8.29349C11.9672 8.29349 11.7068 8.55382 11.7068 8.94431V11.5476C11.7068 11.938 11.4465 12.1984 11.056 12.1984H1.94463C1.55414 12.1984 1.29381 11.938 1.29381 11.5476V8.94431C1.29381 8.55382 1.03349 8.29349 0.643001 8.29349C0.252513 8.29349 -0.0078125 8.55382 -0.0078125 8.94431V11.5476C-0.0078125 12.6539 0.838245 13.5 1.94463 13.5H11.056C12.1624 13.5 13.0085 12.6539 13.0085 11.5476V8.94431C13.0085 8.55382 12.7481 8.29349 12.3576 8.29349Z" fill="' . $this->options[ $this->name_prefix . 'main_color' ] . '"/>
		</svg>';
		
		$groups = $wpdb->get_results($sql, 'ARRAY_A');

		$pd = Personal_Dictionary_Data::get_pd_globsettings();
		$this->options = Personal_Dictionary_Data::get_pd_validated_data_from_array( $pd );
		$group_name = $this->options[ $this->name_prefix . 'change_group_name' ];

		if ($user_id != 0) {

			$content[] = '<div class="' . $this->html_class_prefix . 'import-tab '. $this->html_class_prefix . 'tab-content'. '"  data-function="ays_import_pd">';

			$content[] = '<form method="post" enctype="multipart/form-data">';

			$content[] = '<div class="' . $this->html_class_prefix . 'import-tab-item ' . $this->html_class_prefix . 'import-select-file" >';
				$content[] = '<div class="' . $this->html_class_prefix . 'import-select-file-box" >';
					$content[] = $uploadIcon;
					$content[] = '<div class="' . $this->html_class_prefix . 'import-select-file-box-text" >';
					$content[] = __( "Choose a file", $this->plugin_name );
					$content[] = '</div>';
				$content[] = '</div>';
				$content[] = '<input type="file" name="' . $this->html_class_prefix . 'import-file'. '" id="' . $this->html_class_prefix . 'import_file" accept=".xlsx"/>';
			$content[] = '</div>';

			$content[] = '<div class="' . $this->html_class_prefix . 'import-tab-item ' . $this->html_class_prefix . 'import-select-group-block '. '" >';
			$content[] = '<select class="' . $this->html_class_prefix . 'import-select'. '" name="' . $this->html_class_prefix . 'import-select'. '" >';
			$content[] = '<option value="new_group">' .  __('New ' . $group_name, $this->plugin_name) . '</option>';
			$content[] = '<option disabled>___________</option>';
			foreach ($groups as $key => $value) {
				$content[] = '<option value="'.$value['id'].'">' .  __( $value['name'] , $this->plugin_name) . '</option>';

			}
			$content[] = '</select>';
			$content[] = '</div>';

			$content[] = '<div class="' . $this->html_class_prefix . 'import-tab-item ' . $this->html_class_prefix . 'import-save-button" >';
				$content[] = '<input type="submit" name="' . $this->html_class_prefix . 'import-save-btn'. '" value="Import now" id="' . $this->html_class_prefix . 'import-save-btn'. '" disabled />';
			$content[] = '</div>';

			$content[] = '<div class="' . $this->html_class_prefix . 'import-tab-divider" >';
				$content[] = '<div class="' . $this->html_class_prefix . 'import-tab-divider-row"></div>';
				$content[] = '<div class="' . $this->html_class_prefix . 'import-tab-divider-text">' . __( "Or", $this->plugin_name ) . '</div>';
				$content[] = '<div class="' . $this->html_class_prefix . 'import-tab-divider-row"></div>';
			$content[] = '</div>';

			$content[] = '<div class="' . $this->html_class_prefix . 'import-item-example-container">';
				$content[] = '<div class="' . $this->html_class_prefix . 'import-item-example ' . $this->html_class_prefix . 'secondary-input-fields">';
					$content[] = '<a href="' . PERSONAL_DICTIONARY_PUBLIC_URL .'/partials/export_example/export_example.xlsx" download="export_example.xlsx" type="button">';
						$content[] = $exportIcon;
						$content[] = '<span>' . __( "Export example", $this->plugin_name ) . '</span>';
					$content[] = '</a>';
				$content[] = '</div>';
			$content[] = '</div>';

			$content[] = '</form>';	

			$content[] = '</div>';
			$content[] =  '<div class="ays-pd-preloader">
					<img class="loader" src="'. PERSONAL_DICTIONARY_PUBLIC_URL .'/images/loaders/3-1.svg">
				</div>';

		}else{
			$content[] = "<p style='text-align: center;font-style:italic;'>" . __( "You must log in to see content.", $this->plugin_name ) . "</p>";
		}
		$content = implode( '',$content );

		return $content;
	}

	public function get_custom_css() {

        $content = array();
        if( $this->options[ $this->name_prefix . 'custom_css' ] != '' ){

            $content[] = '<style type="text/css">';
            
	        	$content[] = $this->options[ $this->name_prefix . 'custom_css' ];
            
            $content[] = '</style>';
            
        }

        $content = implode( '', $content );

    	return $content;
    }

	public function get_encoded_options () {
        
        $content = array();

		$content[] = '<script type="text/javascript">';

		$content[] = "
				if(typeof aysPdOptions === 'undefined'){
					var aysPdOptions = [];
				}
				aysPdOptions  = '" . base64_encode( json_encode( $this->options ) ) . "';";
		
		$content[] = '</script>';
            
        $content = implode( '', $content );

    	return $content;
    }

	public function get_styles(){
		
		$content = array();

		// generate styles for hiding/showing menu icons
		if ($this->options[ $this->name_prefix . 'menu_items' ] == 'text') {
			$menu_item_styles = '
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu .' . $this->html_class_prefix . 'nav-menu-item .' . $this->html_class_prefix . 'nav-tab img {
				display: none !important;
			}';
		} else if ($this->options[ $this->name_prefix . 'menu_items' ] == 'icon') {
			$menu_item_styles = '
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu .' . $this->html_class_prefix . 'nav-menu-item .' . $this->html_class_prefix . 'nav-tab span {
				display: none !important;
			}
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu .' . $this->html_class_prefix . 'nav-menu-item .' . $this->html_class_prefix . 'nav-tab img {
				margin-right: 0 !important;
			}';
		} else {
			$menu_item_styles = '
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu .' . $this->html_class_prefix . 'nav-menu-item .' . $this->html_class_prefix . 'nav-tab {
				justify-content: flex-start !important;
			}';
		}

		// generate styles for hiding/showing menu icons for mobile devices
		if ($this->options[ $this->name_prefix . 'menu_items_mobile' ] == 'text') {
			$menu_item_styles_mobile = '
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu .' . $this->html_class_prefix . 'nav-menu-item .' . $this->html_class_prefix . 'nav-tab span {
				display: block !important;
			}
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu .' . $this->html_class_prefix . 'nav-menu-item .' . $this->html_class_prefix . 'nav-tab img {
				display: none !important;
			}';
		} else if ($this->options[ $this->name_prefix . 'menu_items_mobile' ] == 'icon') {
			$menu_item_styles_mobile = '
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu .' . $this->html_class_prefix . 'nav-menu-item .' . $this->html_class_prefix . 'nav-tab span,
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu .' . $this->html_class_prefix . 'nav-menu-item .' . $this->html_class_prefix . 'nav-tab img.' . $this->html_class_prefix . 'nav-tab-image-white {
				display: none !important;
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu .' . $this->html_class_prefix . 'nav-menu-item .' . $this->html_class_prefix . 'nav-tab img.' . $this->html_class_prefix . 'nav-tab-image-black {
				display:block !important;
			}
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu .' . $this->html_class_prefix . 'nav-menu-item .' . $this->html_class_prefix . 'nav-tab.' . $this->html_class_prefix . 'nav-tab-active img.' . $this->html_class_prefix . 'nav-tab-image-white {
				display: block !important;
				margin-right: 0 !important;
			}
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu .' . $this->html_class_prefix . 'nav-menu-item .' . $this->html_class_prefix . 'nav-tab.' . $this->html_class_prefix . 'nav-tab-active img.' . $this->html_class_prefix . 'nav-tab-image-black {
				display: none !important;
			}
			';
		} else {
			$menu_item_styles_mobile = '
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu .' . $this->html_class_prefix . 'nav-menu-item .' . $this->html_class_prefix . 'nav-tab span {
				display: block !important;
			}
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu .' . $this->html_class_prefix . 'nav-menu-item .' . $this->html_class_prefix . 'nav-tab img.' . $this->html_class_prefix . 'nav-tab-image-white {
				display: none !important;
			}
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu .' . $this->html_class_prefix . 'nav-menu-item .' . $this->html_class_prefix . 'nav-tab img.' . $this->html_class_prefix . 'nav-tab-image-black,
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu .' . $this->html_class_prefix . 'nav-menu-item .' . $this->html_class_prefix . 'nav-tab.' . $this->html_class_prefix . 'nav-tab-active img.' . $this->html_class_prefix . 'nav-tab-image-white {
				display:block !important;
			}
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu .' . $this->html_class_prefix . 'nav-menu-item .' . $this->html_class_prefix . 'nav-tab.' . $this->html_class_prefix . 'nav-tab-active img.' . $this->html_class_prefix . 'nav-tab-image-black {
				display: none !important;
			}
			';
		}

		$ays_pd_container_box_shadow = '';
		if ($this->options[ $this->name_prefix . 'enable_container_box_shadow']) {
			$box_shadow_offsets = $this->options[ $this->name_prefix . 'container_box_shadow_x_offset' ] . 'px ' . $this->options[ $this->name_prefix . 'container_box_shadow_y_offset' ] . 'px ' . $this->options[ $this->name_prefix . 'container_box_shadow_z_offset' ] . 'px ';
			$ays_pd_container_box_shadow = 'box-shadow:'.$box_shadow_offsets.$this->options[ $this->name_prefix . 'container_box_shadow_color' ];
		}
		$ays_pd_answer_box_shadow = '';
		if ($this->options[ $this->name_prefix . 'enable_answer_box_shadow']) {
			$answer_box_shadow_offsets = $this->options[ $this->name_prefix . 'answer_box_shadow_x_offset' ] . 'px ' . $this->options[ $this->name_prefix . 'answer_box_shadow_y_offset' ] . 'px ' . $this->options[ $this->name_prefix . 'answer_box_shadow_z_offset' ] . 'px ';
			$ays_pd_answer_box_shadow = $answer_box_shadow_offsets.$this->options[ $this->name_prefix . 'answer_box_shadow_color' ];
		}
        $content[] = '<style type="text/css">';
		

       
        $content[] = ' 
			input[type="search"]:focus{
				box-shadow: unset !important;
			}

			.ays-pd-icon-main-color-fill {
				fill: ' . $this->options[ $this->name_prefix . 'main_color' ] . ';
			}

			div#' . $this->html_class_prefix . 'box_id input[type="button"], div#' . $this->html_class_prefix . 'box_id input[type="submit"],
			div#' . $this->html_class_prefix . 'box_id button.' . $this->html_class_prefix . 'games-write-translation-check-answer {
                font-size: ' . $this->options[ $this->name_prefix . 'button_font_size' ] . 'px;
                border-radius: ' . $this->options[ $this->name_prefix . 'buttons_border_radius' ] . 'px; 
                padding: ' . $this->options[ $this->name_prefix . 'buttons_top_bottom_padding' ] . 'px ' . $this->options[ $this->name_prefix . 'buttons_left_right_padding' ] . 'px; 
            }

			div#' . $this->html_class_prefix . 'box_id input[type="button"]:not(.' . $this->html_class_prefix . 'secondary-input-fields) , input[type="submit"] {
				background-color: ' . $this->options[ $this->name_prefix . 'main_color' ] . ';
                color: ' . $this->options[ $this->name_prefix . 'button_text_color' ] .';
				border: 1px solid ' . $this->options[ $this->name_prefix . 'main_color' ] . ' !important;
				background-image: unset;
			}

			div#' . $this->html_class_prefix . 'box_id input[type="button"].' . $this->html_class_prefix . 'secondary-input-fields {
				color: ' . $this->options[ $this->name_prefix . 'main_color' ] . ';
				border: 1px solid ' . $this->options[ $this->name_prefix . 'main_color' ] . ' !important;
				background-color: ' . $this->options[ $this->name_prefix . 'button_text_color' ] .';
				background-image: unset;
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content,
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu {
				' . $ays_pd_container_box_shadow . '
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu,
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content {
				background-color: ' . $this->options[ $this->name_prefix . 'bg_color' ] . ';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu .' . $this->html_class_prefix . 'nav-menu-item .' . $this->html_class_prefix . 'nav-tab.' . $this->html_class_prefix . 'nav-tab-active {
				background-color: ' . $this->options[ $this->name_prefix . 'main_color' ] . ';
            }

			'.
			$menu_item_styles
			.'
            div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'header-wrap div.' . $this->html_class_prefix . 'header-main div.' . $this->html_class_prefix . 'header-btn-box {
				background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color' ] , 0.2 ) . ';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'header-wrap div.' . $this->html_class_prefix . 'header-full-screen div.' . $this->html_class_prefix . 'open-full-screen a {
				background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color' ] , 0.2 ) . ';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'empty-groups-content div.' . $this->html_class_prefix . 'add_group_btn,
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'empty-groups-content div.' . $this->html_class_prefix . 'add_word_btn,
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'add-group-word-buttons-container div.' . $this->html_class_prefix . 'add_group_btn {
				background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color' ] , 0.2 ) . ';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'each_group_item {
				background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color' ] , 0.2 ) . ';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab a.' . $this->html_class_prefix . 'group-delete-button-item,
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'word-save-and-delete-button-block span {
				color: ' . $this->options[ $this->name_prefix . 'main_color' ] . ' !important;
				border: 1px solid ' . $this->options[ $this->name_prefix . 'main_color' ] . ' !important;
				background-color: ' . $this->options[ $this->name_prefix . 'button_text_color' ] .' !important;
				border-radius: ' . $this->options[ $this->name_prefix . 'buttons_border_radius' ] . 'px !important; 
				padding: ' . $this->options[ $this->name_prefix . 'buttons_top_bottom_padding' ] . 'px ' . $this->options[ $this->name_prefix . 'buttons_left_right_padding' ] . 'px !important;
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'words-each-item-block {
				background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color' ] , 0.2 ) . ';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'save-groups-block {
				background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color' ] , 0.2 ) . ';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-choosing-type label {
				background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color' ] , 0.2 ) . ';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-choosing-type label div.' . $this->html_class_prefix . 'game-type-item-title{
				color: ' . $this->options[ $this->name_prefix . 'main_color' ] .';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'group-tab-words {
				background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color' ] , 0.2 ) . ';
				color: ' . $this->options[ $this->name_prefix . 'text_color' ] .';
			}

			div.' . $this->html_class_prefix . 'group-tab-words  input[type="text"].' . $this->html_class_prefix . 'word_saving_fields {
				border-bottom: 1px solid ' . $this->options[ $this->name_prefix . 'main_color' ] .';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content .' . $this->html_class_prefix . 'games-type-content-settings,
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'flashcards-settings-content {
				background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color' ] , 0.2 ) . ';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-check-words-count label.' . $this->html_class_prefix .'form-check-label,
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'flashcards-check-words-count label.' . $this->html_class_prefix .'form-check-label {
				border: 1px solid ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color' ] , 0.2 ) .';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-check-words-count label.' . $this->html_class_prefix .'form-check-label-active,
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'flashcards-check-words-count label.' . $this->html_class_prefix .'form-check-label-active {
				border: 1px solid ' . $this->options[ $this->name_prefix . 'main_color' ] .';
			}
			
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-check-words-count label.' . $this->html_class_prefix .'form-check-label input[type="radio"],
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'flashcards-check-words-count label.' . $this->html_class_prefix .'form-check-label input[type="radio"] {
				accent-color: ' . $this->options[ $this->name_prefix . 'main_color' ] .';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'flashcards-tab div.' . $this->html_class_prefix .'flashcards-main-container .' . $this->html_class_prefix .'flashcard-each-word-box {
				background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color' ] , 0.2 ) . ';
				color: ' . $this->options[ $this->name_prefix . 'text_color' ] .';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'flashcards-tab div.' . $this->html_class_prefix .'flashcards-main-container .' . $this->html_class_prefix .'flashcard-finish-congratulation-box p {
				color: ' . $this->options[ $this->name_prefix . 'text_color' ] .';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'flashcards-tab div.' . $this->html_class_prefix .'flashcards-main-container button.' . $this->html_class_prefix .'flashcard-finish-restart-button,
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'flashcards-tab div.' . $this->html_class_prefix .'flashcards-main-container button.' . $this->html_class_prefix .'flashcard-finish-restart-from-zero-button,
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'flashcards-tab div.' . $this->html_class_prefix .'flashcards-main-container button.' . $this->html_class_prefix .'flashcard-finish-return-to-settings-button {
				background-color: ' . $this->options[ $this->name_prefix . 'main_color' ] .';
				color: ' . $this->options[ $this->name_prefix . 'button_text_color' ] .';
                font-size: ' . $this->options[ $this->name_prefix . 'button_font_size' ] . 'px;
				border-radius: ' . $this->options[ $this->name_prefix . 'buttons_border_radius' ] . 'px !important; 
				padding: ' . $this->options[ $this->name_prefix . 'buttons_top_bottom_padding' ] . 'px ' . $this->options[ $this->name_prefix . 'buttons_left_right_padding' ] . 'px !important;
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-words-learned-percentage label.' . $this->html_class_prefix .'form-input-label input[type="number"]:focus-visible {
				border: 1px solid ' . $this->options[ $this->name_prefix . 'main_color' ] .';
				color: ' . $this->options[ $this->name_prefix . 'main_color' ] .';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'games-type-content-game-box,
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'matching-words-parent {
				background-color: ' . $this->options[ $this->name_prefix . 'bg_color'] . ';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'games-logo-block {
				background-color: ' . $this->options[ $this->name_prefix . 'bg_color'] . ';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'games-type-content-game-box  div.' . $this->html_class_prefix . 'game-fields label {
				border: 1px solid ' . $this->options[ $this->name_prefix . 'bg_color' ] .';	
				font-size: ' . $this->options[ $this->name_prefix . 'answer_font_size' ] . 'px;
				background-color: ' . $this->options[ $this->name_prefix . 'answer_bg_color' ] . ';
				color: ' . $this->options[ $this->name_prefix . 'text_color' ] .';				
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'games-type-content-game-box  div.' . $this->html_class_prefix . 'game-fields {
				box-shadow: ' . $ays_pd_answer_box_shadow .';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'games-type-content-game-box  div.' . $this->html_class_prefix . 'game-fields	 .checked_answer {
				border: 1px solid ' . $this->options[ $this->name_prefix . 'main_color' ] .';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'game-fields:not(.correct_div) .no_selected {
				border: 1px solid ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color' ] , 0.2 ) .' !important;
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'matching-words-parent label.' . $this->html_class_prefix . 'matching-words-label:not(.' . $this->html_class_prefix . 'matching-game-results) {
				background-color:' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color' ] , 0.2 ) .';
				border: 1px solid ' . $this->options[ $this->name_prefix . 'bg_color' ] .';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'matching-words-parent label.' . $this->html_class_prefix . 'matching-words-label.ui-sortable-handle:not(.' . $this->html_class_prefix . 'matching-game-results):hover {
				border: 1px solid ' . $this->options[ $this->name_prefix . 'main_color' ] .';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'each_group_item .ays-pd_each_group_name {
				color: ' . $this->options[ $this->name_prefix . 'text_color' ] .';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'add_word_button_under_words_list>div {
				border: 1px solid ' . $this->options[ $this->name_prefix . 'main_color' ] .';
				border-radius: ' . $this->options[ $this->name_prefix . 'buttons_border_radius' ] . 'px;
				padding: ' . $this->options[ $this->name_prefix . 'buttons_top_bottom_padding' ] . 'px ' . $this->options[ $this->name_prefix . 'buttons_left_right_padding' ] . 'px;
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'add_word_button_under_words_list>div>span {
				color: ' . $this->options[ $this->name_prefix . 'main_color' ] .' !important;
			}

			div#' . $this->html_class_prefix . 'box_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'close-button-div-word-top {
				border-bottom: 1px solid ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color' ] , 0.2 ) .';
			}
			.' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'words-each-item-block div.' . $this->html_class_prefix . 'group-tab-edit-layer div.' . $this->html_class_prefix . 'word-translation-fields-parent input {
				border-bottom: 1px solid ' . $this->options[ $this->name_prefix . 'main_color' ] .';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'each_group_item .ays-pd_each_group_title_words_count {
				color: ' . $this->options[ $this->name_prefix . 'text_color' ] .';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'words-each-item-block span.' . $this->html_class_prefix . 'each_word_span,
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'words-each-item-block span.' . $this->html_class_prefix . 'each_translation,
			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'words-each-item-block p.' . $this->html_class_prefix . 'each_word_number {
				color: ' . $this->options[ $this->name_prefix . 'text_color' ] .';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'each_group_item div.' . $this->html_class_prefix . 'dropdown-buttons button:hover {
				background-color: ' .  Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color' ], 0.2 ) . ';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'groups-modal-main-container div.' . $this->html_class_prefix . 'groups-modal-content div.' . $this->html_class_prefix . 'groups-modal-header{
				background-color: ' .  $this->options[ $this->name_prefix . 'main_color' ] . ';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'groups-modal-main-container div.' . $this->html_class_prefix . 'groups-modal-content div.' . $this->html_class_prefix . 'groups-modal-header span{
				color: #fff;
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'groups-modal-main-container div.' . $this->html_class_prefix . 'groups-modal-content div#' . $this->html_class_prefix . 'groups-modal-body{
				background-color: ' . $this->options[ $this->name_prefix . 'bg_color' ] . ';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'groups-modal-main-container div.' . $this->html_class_prefix . 'groups-modal-content div#' . $this->html_class_prefix . 'groups-modal-body div.ays-pd-each_group_item-move{
				background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color' ] , 0.2 ) . ';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'groups-modal-main-container div.' . $this->html_class_prefix . 'groups-modal-content div#' . $this->html_class_prefix . 'groups-modal-body div.ays-pd-each_group_item-move p.ays-pd_each_group_name-move{
				color: ' . $this->options[ $this->name_prefix . 'text_color' ] .';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'confirm-modal-main-container div.' . $this->html_class_prefix . 'confirm-modal-content{
				background-color: ' . $this->options[ $this->name_prefix . 'bg_color' ] . ';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'confirm-modal-main-container div.' . $this->html_class_prefix . 'confirm-modal-header p{
				color: ' . $this->options[ $this->name_prefix . 'text_color' ] .';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'confirm-modal-main-container button.' . $this->html_class_prefix . 'confirm-modal-submit{
				background-color: ' .  $this->options[ $this->name_prefix . 'main_color' ] . ';
				color: ' . $this->options[ $this->name_prefix . 'button_text_color' ] .';
				font-size: ' . $this->options[ $this->name_prefix . 'button_font_size' ] . 'px; 
				border-radius: ' . $this->options[ $this->name_prefix . 'buttons_border_radius' ] . 'px; 
				padding: ' . $this->options[ $this->name_prefix . 'buttons_top_bottom_padding' ] . 'px ' . $this->options[ $this->name_prefix . 'buttons_left_right_padding' ] . 'px;
			}

			div#' . $this->html_class_prefix . 'box_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-tab .ays-pd-games-question-para {
				font-size: ' . $this->options[ $this->name_prefix . 'question_font_size' ] . 'px;
				color: ' . $this->options[ $this->name_prefix . 'text_color' ] .';
			}

			div#' . $this->html_class_prefix . 'box_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-tab .ays-pd-games-find-translation{
				font-size: ' . $this->options[ $this->name_prefix . 'question_font_size' ] . 'px;
				color: ' . $this->options[ $this->name_prefix . 'text_color' ] .';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'games-type-content-game-box div.' . $this->html_class_prefix . 'games-write-translation-answer-box:hover {
				background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color' ] , 0.2 ) . ';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'games-type-content-game-box div.' . $this->html_class_prefix . 'games-write-translation-answer-box:hover button.' . $this->html_class_prefix . 'games-write-translation-check-answer {
				box-shadow: 0 0 0 2px white, 0 0 0 3px ' .  $this->options[ $this->name_prefix . 'main_color' ] . ';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'games-type-content-game-box button.' . $this->html_class_prefix . 'games-write-translation-check-answer {
				background-color: ' .  $this->options[ $this->name_prefix . 'main_color' ] . ';
			}

			// div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'games-type-content-game-box button.' . $this->html_class_prefix . 'games-write-translation-check-answer:hover {
			// 	box-shadow: 0 0 0 2px white, 0 0 0 3px ' .  $this->options[ $this->name_prefix . 'main_color' ] . ';
			// }


			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id input[type="submit"]:hover {
				background-color: ' . $this->options[ $this->name_prefix . 'main_color' ] . ';
				color: ' . $this->options[ $this->name_prefix . 'button_text_color' ] .';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'import-tab {
				border: 1px dashed ' .  Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color' ] , 0.2 ) . ';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'import-tab div.' . $this->html_class_prefix . 'import-select-file-box-text {
				color:' .  $this->options[ $this->name_prefix . 'main_color' ] . ';
			}

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'import-tab div.' . $this->html_class_prefix . 'import-item-example a {
				color: ' . $this->options[ $this->name_prefix . 'main_color' ] . ';
				border: 1px solid ' . $this->options[ $this->name_prefix . 'main_color' ] . ';
				border-radius: ' . $this->options[ $this->name_prefix . 'buttons_border_radius' ] . 'px; 
				padding: ' . $this->options[ $this->name_prefix . 'buttons_top_bottom_padding' ] . 'px ' . $this->options[ $this->name_prefix . 'buttons_left_right_padding' ] . 'px;
			}

            ';

			$content[] = $this->get_css_mobile_part($menu_item_styles_mobile);

			if( $this->options[ $this->name_prefix . 'container_border_radius'] ){
				$content[] = '
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content,
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu {
					border-radius: ' . $this->options[ $this->name_prefix . 'container_border_radius' ] . 'px;
	            }
				';
			}

			if( $this->options[ $this->name_prefix . 'container_border_width'] ){
				$content[] = '
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content,
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu {
					border-width: ' . $this->options[ $this->name_prefix . 'container_border_width' ] . 'px;
	            }
				';
			}

			if( $this->options[ $this->name_prefix . 'container_border_color'] ){
				$content[] = '
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content,
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu {
					border-color: ' . $this->options[ $this->name_prefix . 'container_border_color' ] . ';
	            }
				';
			}

			if( $this->options[ $this->name_prefix . 'container_border_style'] ){
				$content[] = '
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content,
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu {
					border-style: ' . $this->options[ $this->name_prefix . 'container_border_style' ] . ';
	            }';
			}

			if( $this->options[ $this->name_prefix . 'enable_answer_border'] == 'on' ){
				$answ_border_width = $this->options[ $this->name_prefix . 'answer_border_width'] ? $this->options[ $this->name_prefix . 'answer_border_width'] : 1;
				$answ_border_color = $this->options[ $this->name_prefix . 'answer_border_color'] ? $this->options[ $this->name_prefix . 'answer_border_color'] : 'rgba(148,148,148,0.85)';
				$answ_border_style = $this->options[ $this->name_prefix . 'answer_border_style'] ? $this->options[ $this->name_prefix . 'answer_border_style'] : 'solid';
				$content[] = '

				.' . $this->html_class_prefix . 'box div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'game-fields  {
	                border: '.$answ_border_width.'px '.$answ_border_style.' '.$answ_border_color.';
	            }
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'games-type-content-game-box  div.' . $this->html_class_prefix . 'game-fields:hover {
					border: '.$answ_border_width.'px solid ' . $this->options[ $this->name_prefix . 'main_color' ] .';
				}';
			}else{
				$content[] = '
					div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'games-type-content-game-box  div.' . $this->html_class_prefix . 'game-fields:hover {
						border: 1px solid ' . $this->options[ $this->name_prefix . 'main_color' ] .' !important;
					}';
			}

			$ays_pd_title_text_shadow = '';
	        if ($this->options[ $this->name_prefix . 'enable_title_text_shadow']) {
        		$text_shadow_offsets = $this->options[ $this->name_prefix . 'title_text_shadow_x_offset' ] . 'px ' . $this->options[ $this->name_prefix . 'title_text_shadow_y_offset' ] . 'px ' . $this->options[ $this->name_prefix . 'title_text_shadow_z_offset' ] . 'px ';
	            $ays_pd_title_text_shadow = 'text-shadow:'.$text_shadow_offsets.$this->options[ $this->name_prefix . 'title_text_shadow_color' ];
	        }
			
			// Answer View
			if( $this->options[ $this->name_prefix . 'answer_view'] == 'list' ){
				$content[] = '				
				div.' . $this->html_class_prefix . 'box div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'game-fields-container div.' . $this->html_class_prefix . 'game-fields{
					flex-basis: 90%;
				}
				';
			}

	        $ays_pd_title_alignment = 'text-align:'.$this->options[ $this->name_prefix . 'title_alignment'];

			$ays_pd_title_alignment_mobile = 'text-align:'.$this->options[ $this->name_prefix . 'title_alignment_mobile'];

			$ays_pd_title_font_size = 'font-size:'. $this->options[ $this->name_prefix . 'title_font_size'] . 'px';

			// Show PD Title 
			if($this->options[ $this->name_prefix . 'show_title' ] == 'on'){
				$content[] = 'h2.' . $this->html_class_prefix . 'title{
					display: block;
					'.$ays_pd_title_font_size.';
					'.$ays_pd_title_text_shadow.';
					'.$ays_pd_title_alignment_mobile.';
				}';
			}else if($this->options[ $this->name_prefix . 'show_title' ] == 'off'){
				$content[] = 'h2.' . $this->html_class_prefix . 'title{
					display: none;
				}';
			}
			
			// Show PD Title mobile
			if($this->options[ $this->name_prefix . 'show_title_mobile' ] == 'on'){
				$show_title_mobile = 'h2.' . $this->html_class_prefix . 'title{
					display: block;
					'.$ays_pd_title_font_size.';
					'.$ays_pd_title_text_shadow.';
					'.$ays_pd_title_alignment.';
				}';
			} else if($this->options[ $this->name_prefix . 'show_title_mobile' ] == 'off'){
				$show_title_mobile = 'h2.' . $this->html_class_prefix . 'title{
					display: none;				
				}';
			}

		

			$ays_pd_description_font_size = 'font-size:'. $this->options[ $this->name_prefix . 'description_font_size'] . 'px';
			
			$ays_pd_description_alignment = 'text-align:'. $this->options[ $this->name_prefix . 'desc_alignment'];
			
			$ays_pd_description_alignment_mobile = 'text-align:'.$this->options[ $this->name_prefix . 'desc_alignment_mobile'];

			$ays_pd_description_text_shadow = '';
	        if ($this->options[ $this->name_prefix . 'enable_description_text_shadow']) {
        		$description_text_shadow_offsets = $this->options[ $this->name_prefix . 'description_text_shadow_x_offset' ] . 'px ' . $this->options[ $this->name_prefix . 'description_text_shadow_y_offset' ] . 'px ' . $this->options[ $this->name_prefix . 'description_text_shadow_z_offset' ] . 'px ';
	            $ays_pd_description_text_shadow = 'text-shadow:'.$description_text_shadow_offsets.$this->options[ $this->name_prefix . 'description_text_shadow_color' ];
	        }

 			// Show PD Description
			if($this->options[ $this->name_prefix . 'show_description' ] == 'on'){
				$content[] = 'div.' . $this->html_class_prefix . 'description{
					'.$ays_pd_description_font_size.';
					'.$ays_pd_description_alignment.';
					'.$ays_pd_description_text_shadow.';
				}';
			}else if($this->options[ $this->name_prefix . 'show_description' ] == 'off'){
				$content[] = 'div.' . $this->html_class_prefix . 'description{
					display: none;				
				}';
			}

			// Show PD Description mobile
			if($this->options[ $this->name_prefix . 'show_description_mobile' ] == 'on'){
				$show_description_mobile = 'div.' . $this->html_class_prefix . 'description{
					display: block;
					'.$ays_pd_description_font_size.';
					'.$ays_pd_description_alignment_mobile.';
					'.$ays_pd_description_text_shadow.';
				}';
			}else if($this->options[ $this->name_prefix . 'show_description_mobile' ] == 'off'){
				$show_description_mobile = 'div.' . $this->html_class_prefix . 'description{
					display: none;				
				}';
			}

			$content[] = '@media screen and (max-width: 768px){
				'.$show_title_mobile.'
				'.$show_description_mobile.'
			}';

    	$content[] = '</style>';

    	$content = implode( '', $content );

    	return $content;
    }
	public function get_css_mobile_part($menu_item_styles_mobile) {
		$content = '';

		if( $this->options[ $this->name_prefix . 'enable_answer_border'] == 'on' ){
			$answ_border_width_mobile = $this->options[ $this->name_prefix . 'answer_border_width_mobile'] ? $this->options[ $this->name_prefix . 'answer_border_width_mobile'] : 1;			
			$answ_border_style = $this->options[ $this->name_prefix . 'answer_border_style'] ? $this->options[ $this->name_prefix . 'answer_border_style'] : 'solid';
			$answ_border_color = $this->options[ $this->name_prefix . 'answer_border_color'] ? $this->options[ $this->name_prefix . 'answer_border_color'] : 'rgba(148,148,148,0.85)';

			$answer_border_mobile_styles = '
			.' . $this->html_class_prefix . 'box div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'game-fields {                
	            border: '. $answ_border_width_mobile .'px '. $answ_border_style .' ' . $answ_border_color .' !important;
            }

			div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'games-type-content-game-box  div.' . $this->html_class_prefix . 'game-fields:hover {
				border: '.$answ_border_width_mobile.'px solid ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] .' !important;
            }
			';
		}else{
			$answer_border_mobile_styles = '
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'games-type-content-game-box  div.' . $this->html_class_prefix . 'game-fields:hover {
					border: 1px solid ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] .' !important;
				}
				';
		}

		$content .= '
			@media screen and (max-width: 768px){

				.ays-pd-icon-main-color-fill {
					fill: ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] . ';
				}

				div#' . $this->html_class_prefix . 'box_id input[type="button"]:not(.' . $this->html_class_prefix . 'secondary-input-fields) , input[type="submit"] {
					background-color: ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] . ';
					border: 1px solid ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] . ' !important;
				}

				div#' . $this->html_class_prefix . 'box_id input[type="button"].' . $this->html_class_prefix . 'secondary-input-fields {
					color: ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] . ';
					border: 1px solid ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] . ' !important;
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu .' . $this->html_class_prefix . 'nav-menu-item .' . $this->html_class_prefix . 'nav-tab.' . $this->html_class_prefix . 'nav-tab-active {
					background-color: ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] . ';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'header-wrap div.' . $this->html_class_prefix . 'header-main div.' . $this->html_class_prefix . 'header-btn-box {
					background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color_mobile' ] , 0.2 ) . ';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'header-wrap div.' . $this->html_class_prefix . 'header-full-screen div.' . $this->html_class_prefix . 'open-full-screen a {
					background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color_mobile' ] , 0.2 ) . ';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'empty-groups-content div.' . $this->html_class_prefix . 'add_group_btn,
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'empty-groups-content div.' . $this->html_class_prefix . 'add_word_btn,
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'add-group-word-buttons-container div.' . $this->html_class_prefix . 'add_group_btn {
					background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color_mobile' ] , 0.2 ) . ';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'each_group_item {
					background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color_mobile' ] , 0.2 ) . ';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab a.' . $this->html_class_prefix . 'group-delete-button-item,
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'word-save-and-delete-button-block span {
					color: ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] . ' !important;
					border: 1px solid ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] . ' !important;
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'words-each-item-block {
					background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color_mobile' ] , 0.2 ) . ';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'save-groups-block {
					background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color_mobile' ] , 0.2 ) . ';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-choosing-type label {
					background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color_mobile' ] , 0.2 ) . ';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-choosing-type label div.' . $this->html_class_prefix . 'game-type-item-title{
					color: ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] .';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'group-tab-words {
					background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color_mobile' ] , 0.2 ) . ';
					color: ' . $this->options[ $this->name_prefix . 'text_color_mobile' ] .';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'each_group_item .ays-pd_each_group_name {
					color: ' . $this->options[ $this->name_prefix . 'text_color_mobile' ] .';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'each_group_item .ays-pd_each_group_title_words_count {
					color: ' . $this->options[ $this->name_prefix . 'text_color_mobile' ] .';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'words-each-item-block span.' . $this->html_class_prefix . 'each_word_span,
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'words-each-item-block span.' . $this->html_class_prefix . 'each_translation,
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'words-each-item-block p.' . $this->html_class_prefix . 'each_word_number {
					color: ' . $this->options[ $this->name_prefix . 'text_color_mobile' ] .';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'groups-modal-main-container div.' . $this->html_class_prefix . 'groups-modal-content div#' . $this->html_class_prefix . 'groups-modal-body div.ays-pd-each_group_item-move p.ays-pd_each_group_name-move{
					color: ' . $this->options[ $this->name_prefix . 'text_color_mobile' ] .';
				}
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'confirm-modal-main-container div.' . $this->html_class_prefix . 'confirm-modal-header p{
					color: ' . $this->options[ $this->name_prefix . 'text_color_mobile' ] .';
				}

				div.' . $this->html_class_prefix . 'group-tab-words  input[type="text"].' . $this->html_class_prefix . 'word_saving_fields {
					border-bottom: 1px solid ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] .';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content .' . $this->html_class_prefix . 'games-type-content-settings {
					background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color_mobile' ] , 0.2 ) . ';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-check-words-count label.' . $this->html_class_prefix .'form-check-label {
					border: 1px solid ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color_mobile' ] , 0.2 ) .';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-check-words-count label.' . $this->html_class_prefix .'form-check-label-active {
					border: 1px solid ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] .';
				}
				
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-check-words-count label.' . $this->html_class_prefix .'form-check-label input[type="radio"] {
					accent-color: ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] .';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-words-learned-percentage label.' . $this->html_class_prefix .'form-input-label input[type="number"]:focus-visible {
					border: 1px solid ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] .';
					color: ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] .';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'games-type-content-game-box  div.' . $this->html_class_prefix . 'game-fields	 .checked_answer {
					border: 1px solid ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] .';
				}';

				if( $this->options[ $this->name_prefix . 'answer_view_mobile'] == 'grid' ){
					$content .= '				
					div.' . $this->html_class_prefix . 'box div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'game-fields-container div.' . $this->html_class_prefix . 'game-fields{
						flex-basis: 40% !important;
		            }
					';
				}

				$content .= $answer_border_mobile_styles;

				$content .= '
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'game-fields:not(.correct_div) .no_selected {
					border: 1px solid ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color_mobile' ] , 0.2 ) .' !important;
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'add_word_button_under_words_list>div {
					border: 1px solid ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] .';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'add_word_button_under_words_list>div>span {
					color: ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] .' !important;
				}

				div#' . $this->html_class_prefix . 'box_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'close-button-div-word-top {
					border-bottom: 1px solid ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color_mobile' ] , 0.2 ) .';
				}

				.' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'words-each-item-block div.' . $this->html_class_prefix . 'group-tab-edit-layer div.' . $this->html_class_prefix . 'word-translation-fields-parent input {
					border-bottom: 1px solid ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] .';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content .' . $this->html_class_prefix . 'group-tab div.' . $this->html_class_prefix . 'each_group_item div.' . $this->html_class_prefix . 'dropdown-buttons button:hover {
					background-color: ' .  Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color_mobile' ], 0.2 ) . ';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'groups-modal-main-container div.' . $this->html_class_prefix . 'groups-modal-content div.' . $this->html_class_prefix . 'groups-modal-header{
					background-color: ' .  $this->options[ $this->name_prefix . 'main_color_mobile' ] . ';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'groups-modal-main-container div.' . $this->html_class_prefix . 'groups-modal-content div#' . $this->html_class_prefix . 'groups-modal-body div.ays-pd-each_group_item-move{
					background-color: ' . Personal_Dictionary_Data::hex2rgba( $this->options[ $this->name_prefix . 'main_color_mobile' ] , 0.2 ) . ';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'confirm-modal-main-container button.' . $this->html_class_prefix . 'confirm-modal-submit{
					background-color: ' .  $this->options[ $this->name_prefix . 'main_color_mobile' ] . ';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id input[type="submit"]:hover {
					background-color: ' . $this->options[ $this->name_prefix . 'main_color_mobile' ] . ';
				}

                div#' . $this->html_class_prefix . 'box_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-tab .ays-pd-games-question-para {
					font-size: ' . $this->options[ $this->name_prefix . 'question_font_size_for_mobile' ] . 'px;
					color: ' . $this->options[ $this->name_prefix . 'text_color_mobile' ] .';
				}
				div#' . $this->html_class_prefix . 'box_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-tab .ays-pd-games-find-translation{
					font-size: ' . $this->options[ $this->name_prefix . 'question_font_size_for_mobile' ] . 'px;
					color: ' . $this->options[ $this->name_prefix . 'text_color_mobile' ] .';

				}
				'.

				$menu_item_styles_mobile

				.'
				div#' . $this->html_class_prefix . 'description-id' . ' {
					font-size: ' . $this->options[ $this->name_prefix . 'description_font_size_for_mobile' ] . 'px;
				}

				h2#' . $this->html_class_prefix . 'title-id' . ' {
					font-size: ' . $this->options[ $this->name_prefix . 'title_font_size_for_mobile' ] . 'px;
				}

				div#' . $this->html_class_prefix . 'box_id input[type="button"] {
					font-size: ' . $this->options[ $this->name_prefix . 'button_font_size_for_mobile' ] . 'px; 
            	}

            	div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu,
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content {
					background-color: ' . $this->options[ $this->name_prefix . 'bg_color_mobile' ] . ';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'games-type-content-game-box {
					background-color: ' . $this->options[ $this->name_prefix . 'bg_color_mobile'] . ';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'games-logo-block {
					background-color: ' . $this->options[ $this->name_prefix . 'bg_color_mobile'] . ';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'games-type-content-game-box  div.' . $this->html_class_prefix . 'game-fields label {
					border: 1px solid ' . $this->options[ $this->name_prefix . 'bg_color_mobile' ] .';
					font-size: ' . $this->options[ $this->name_prefix . 'answer_font_size_for_mobile' ] . 'px;
					color: ' . $this->options[ $this->name_prefix . 'text_color_mobile' ] .';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'groups-modal-main-container div.' . $this->html_class_prefix . 'groups-modal-content div#' . $this->html_class_prefix . 'groups-modal-body{
					background-color: ' . $this->options[ $this->name_prefix . 'bg_color_mobile' ] . ';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'confirm-modal-main-container div.' . $this->html_class_prefix . 'confirm-modal-content{
					background-color: ' . $this->options[ $this->name_prefix . 'bg_color_mobile' ] . ';
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content,
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu {
					border-radius: ' . $this->options[ $this->name_prefix . 'container_border_radius_mobile' ] . 'px !important;    
				}

				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'content,
				div#' . $this->html_class_prefix . 'box_id div#' . $this->html_class_prefix . 'container_id div.' . $this->html_class_prefix . 'nav-menu {
					border-width: ' . $this->options[ $this->name_prefix . 'container_border_width_mobile' ] . 'px !important;
					border-style: ' . $this->options[ $this->name_prefix . 'container_border_style_mobile' ] . ' !important;
					border-color: ' . $this->options[ $this->name_prefix . 'container_border_color_mobile' ] . ' !important;
	            }
            }
				
			@media screen and (max-width: 352px){';
				if( $this->options[ $this->name_prefix . 'answer_view_mobile'] == 'grid' ){
					$content .= '				
					div.' . $this->html_class_prefix . 'box div.' . $this->html_class_prefix . 'games-type-content div.' . $this->html_class_prefix . 'game-fields-container div.' . $this->html_class_prefix . 'game-fields{
						flex-basis: 65% !important;
		            }
					';
				}'
			}		
			
		';

        return $content;
	}

    public function ays_pd_move_word_to_group(){
		global $wpdb;
		$words_table 	= esc_sql( $wpdb->prefix . PERSONAL_DICTIONARY_DB_PREFIX . 'words' );
		$pd_group_id = (isset($_REQUEST['group_id']) && $_REQUEST['group_id'] != '') ? sanitize_text_field( $_REQUEST['group_id'] ) : null;
		$pd_word_id  = (isset($_REQUEST['word_id']) && $_REQUEST['word_id'] != '') ? sanitize_text_field( $_REQUEST['word_id'] ) : null;
		$insert_results = 0;
		if(isset($pd_group_id) && isset($pd_word_id)){
			$insert_results = $wpdb->update(
				$words_table,
				array(
					'category_id' => $pd_group_id,
				),
				array( 'id' => $pd_word_id ),
				array(
					'%d', // catgeroy id
				),
				array( '%d' )
			);
		}
		$response = array(
			"status" => true,
			"moved_to_group" => $insert_results,
		);
		ob_end_clean();
		$ob_get_clean = ob_get_clean();
		echo json_encode( $response );
		wp_die();		
	}
}
